const mongoose = require('mongoose');
const bcrypt = require('bcrypt');

const UserSchema = mongoose.Schema({
  name :{
    type: String, default:""
  },

  email :{
    type: String ,
    required: true,
    lowercase: true
  },

  username :{
    type: String,
    required: true,
    lowercase: true
  },

  createdOn :{
    type: Date,
    default: Date.now
  },

  password: {
    type: String,
    required: true
  },

  roles: {
      type: String,
			enum:["client","provider","subclient","admin"],
      required: true,
  },

  rfc: {
      type:String,
      required: true,
  },

  isValid :{
    type: Boolean,
    default: true
  },

  isActive :{
    type: Boolean,
    default: false
  },

  tel: {
    type:String,
    default:""
  },

  referenceToken : {
    type:String,
  },

  activationToken : {
    type:String
  },

  clientId : {
    type: mongoose.Schema.Types.ObjectId,
    ref: 'User'
}
});

// Before Saving run this hook
UserSchema.pre(
  'save',
  async function(next) {
    const user = this;
    if(user.isModified('password')){
      const hash = await bcrypt.hash(user.password, parseInt(process.env.BCRYPT_SALT));
      this.password = hash;
      this.activationToken = Math.floor(1000 + Math.random() * 9000);
    }
    next();
  }

);


// Method to check if password is valid
UserSchema.methods.isValidPassword = async function(password) {
  const user = this;
  const compare = await bcrypt.compare(password, user.password);

  return compare;
};

UserSchema.methods.hashPassword = async function(password) {
  const user = this;
  const hash = await bcrypt.hash(password ? password : user.password, parseInt(process.env.BCRYPT_SALT));
  return hash;
};

module.exports =  mongoose.model('User', UserSchema);