var array = require('lodash/array');

const Mailgun = require('mailgun-js');
const Twig = require('twig');

const objectId = require('mongoose').Types.ObjectId;
const Document = require('./models/documents');
const ZipDocument = require('./models/zipDocument');
const ZipExtractedDocument = require('./models/zipExtractedDocument');
const Directory = require('./models/directory');
const accountsRelations = require('./models/accountsRelations');
const Facturas = require('./models/facturas');
const Notification = require('./models/notification');
const model = require("mongoose").model;

const { s3 } = require('../config/aws');
const fs = require('fs');

// PDF Reader Library
const pdf = require('pdf-parse');

//XML -> JSON convertion
const parseString = require("xml2js").parseString;

const { monthlyDocumentsTemplate, nonRecurringDocumentsTemplate } = require('./utility');
const documents = require('./models/documents');
const User = require('./models/user');
const { cleanUpUpload } = require('../config/uploads');
const { v4: uuidv4 } = require('uuid');

const deleteInformationFromUserModel = function (userData) {

    // Delete these fields from the User-Modal, Only allow Data to be seen by end-user
    delete userData["_id"];
    delete userData["password"];
    delete userData["isValid"];
    delete userData["__v"];
    delete userData["referenceToken"];
    delete userData["activationToken"];

    return userData
}

const sendOTPinMail = function (email, userToken, username = '') {
    const mailgun = new Mailgun({ apiKey: process.env.MAIL_API, domain: process.env.MAIL_DOMAIN });
    const fromEmail = 'no-reply@kentaurus.com.mx';

    // Get Rendered HTML
    Twig.renderFile('./templates/emails/register.html.twig', { email: email, userToken: userToken, username: username }, (err, html) => {

        const data = {
            //Specify email data
            from: fromEmail,
            //The email to contact
            to: email,
            //Subject and text data
            subject: 'Hello from Ebitsa, Verify your Email',
            html: html
        }

        // MailGun Fire Function

        mailgun.messages().send(data, function (err, body) {
            // If there is an error, render the error page
            if (err) {
                console.log("got an error on mailgun: ", err);
                return false;
            }
            // Else we can greet and leave
            else {
                console.log(`Mail Send, To the user ${email}`);
                return true;
            }
        });
    });

}

const SendRegistrationLinkViaEmail = function (email, requestedBy, requestedByEmail, referenceToken, role = 'provider') {
    const mailgun = new Mailgun({ apiKey: process.env.MAIL_API, domain: process.env.MAIL_DOMAIN });
    const fromEmail = 'no-reply@kentaurus.com.mx';

    // Get Rendered HTML
    Twig.renderFile('./templates/emails/signp.html.twig', { email: email, requestedBy: requestedBy, requestedByEmail: requestedByEmail, url: process.env.REGISTER_URL, referenceToken: referenceToken, role: role }, (err, html) => {

        const data = {
            //Specify email data
            from: fromEmail,
            //The email to contact
            to: email,
            //Subject and text data
            subject: `Hi There, ${requestedBy} invited you to join EBITSA`,
            html: html
        }

        // MailGun Fire Function

        mailgun.messages().send(data, function (err, body) {
            // If there is an error, render the error page
            if (err) {
                console.log("got an error on mailgun: ", err);
                return false;
            }
            // Else we can greet and leave
            else {
                console.log(`Mail Send, To the user ${email}`);
                return true;
            }
        });
    });
}

const sendResetOTPinMail = function (email, userToken) {
    const mailgun = new Mailgun({ apiKey: process.env.MAIL_API, domain: process.env.MAIL_DOMAIN });
    const fromEmail = 'no-reply@kentaurus.com.mx';

    // Get Rendered HTML
    Twig.renderFile('./templates/emails/reset.html.twig', { email: email, userToken: userToken }, (err, html) => {

        const data = {
            //Specify email data
            from: fromEmail,
            //The email to contact
            to: email,
            //Subject and text data
            subject: 'Reset your password, EBITSA',
            html: html
        }

        // MailGun Fire Function

        mailgun.messages().send(data, function (err, body) {
            // If there is an error, render the error page
            if (err) {
                console.log("got an error on mailgun: ", err);
                return false;
            }
            // Else we can greet and leave
            else {
                console.log(`Mail Send, To the user ${email}`);
                return true;
            }
        });
    });

}

const createNewDirectory = () => {
    return new Promise((resolve, reject) => {
        try {
            const d = new Date();
            const currentYear = d.getFullYear();
            const currentMonth = d.getMonth() + 1;

            let directory = new Directory();

            directory.documents.reccuring = [
                {
                    year: currentYear,
                    months: [
                        {
                            month: currentMonth,
                            documents: monthlyDocumentsTemplate
                        }
                    ]
                }
            ]
            directory.documents.nonRecurring = nonRecurringDocumentsTemplate

            directory.save().then((directory) => {
                if (!directory) {
                    reject(false);
                }
                console.log("New Directory Created : ", directory._id.toString())

                resolve(directory._id);
            }).catch((err) => {
                // Console the Error
                console.log(err);

                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}

const uploadFilesTos3 = (fileInformation) => {
    return new Promise((resolve, reject) => {
        try {
            fs.readFile(fileInformation.path, (err, fileBody) => {
                if (err) {
                    reject(false);
                } else {
                    let params = {
                        Bucket: process.env.BUCKET_NAME,
                        Key: fileInformation.name,
                        Body: fileBody,
                        ContentType: fileInformation.contentType,
                        ACL: process.env.AWS_ACL
                    };
                    s3.upload(params, (err, result) => {
                        if (err) {
                            reject(false);
                        } else {
                            console.log("\nFILE UPLOADED TO S3.");
                            resolve(result);
                        }
                    })
                }
            });
        } catch (e) {
            reject(false);
        }
    });
}


const getFilesFromS3 = (s3Path, contentType) => {
    return new Promise((resolve, reject) => {
        try {
            let data = {
                Expires: 900,
                Bucket: process.env.BUCKET_NAME,
                Key: s3Path,
                ResponseContentDisposition: 'inline'
            };

            if (contentType == 'application/pdf') data.ResponseContentType = contentType;

            s3.getSignedUrl('getObject', data, (err, url) => {
                if (err) {
                    reject(false);
                } else {
                    resolve(url);
                }
            });
        } catch (e) {
            reject(false);
        }
    });
}

const DocumentCreation = (documentType, responseFromS3, file, documentId, documentData) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Create a New Document
            let doc = new Document();
            doc.name = file.name.replace(/\.[^/.]+$/, "");
            doc.s3Path = responseFromS3.Key;
            doc.contentType = file.type;
            doc.DocumentType = documentType;
            doc.uniqueId = documentId;
            // This is Documentation Parsing, To extract data from the documents XMLs, PDFs etc and then put it in a string for later distribution.
            doc.DocumentData = documentData;
            doc.save().then((documentDetails) => {
                resolve(documentDetails._id);
            }).catch((e) => {
                // console.log(e);
                reject(false);
            })
        } catch (e) {
            // console.log(e);
            reject(false);
        }
    });
}

const AddNotification = (directory_id, year, month, sender, notification_type) => {
    console.log("year", year);
    console.log("month", month);

    return new Promise(async (resolve, reject) => {
        try {
            const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directory_id) }).lean();
            let receiver = "";
            if (DirectoryAccountMap) {
                console.log(DirectoryAccountMap);
                let userdata = DirectoryAccountMap.clientId;
                if ((userdata.toString() == sender)) {
                    receiver = DirectoryAccountMap.providerId;
                } else {
                    receiver = DirectoryAccountMap.clientId;
                }
            }

            const countExistingRecord = await Notification.find({ year: year, month: month, directory: directory_id, sender: sender, receiver: receiver, type: "fileupload" }).count()
            if (countExistingRecord > 0) {
                await Notification.deleteMany({ year: year, month: month, directory: directory_id, sender: sender, receiver: receiver, type: "fileupload" });
                notification_type = "group";
            }
            let notification_data = new Notification();
            notification_data.year = year;
            notification_data.month = month;
            notification_data.directory = directory_id;
            notification_data.sender = sender;
            notification_data.receiver = receiver;
            notification_data.type = notification_type;
            notification_data.is_read = 0;
            notification_data.save().then((documentDetails) => {
                resolve(documentDetails._id);
            }).catch((e) => {
                console.log(e);
                reject(false);
            })
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const DirectoryUpdation = (year, month, directory_id, documentType, isRecurring, documentIds, subDocumentType = "") => {
    return new Promise((resolve, reject) => {
        try {
            Directory.findById(objectId(directory_id)).then((directoryDetails) => {

                // Recurring Or Non Recurring Document
                const travesingType = (isRecurring.toLowerCase() == "true") ? "reccuring" : "nonRecurring";

                if (travesingType == "reccuring") {
                    // This is the Year Index
                    const yearIndex = findIndexOnObject(year, directoryDetails.toJSON().documents[travesingType], "year");

                    // This is the Month Index
                    const monthIndex = findIndexOnObject(month, directoryDetails.toJSON().documents[travesingType][yearIndex].months, "month");

                    // This depends on document Type
                    const documentTypeIndex = findIndexOnObject(documentType, directoryDetails.toJSON().documents[travesingType][yearIndex].months[monthIndex].documents, "name");

                    if (subDocumentType) {
                        const subDocumentTypeIndex = findIndexOnObject(subDocumentType, directoryDetails.toJSON().documents[travesingType][yearIndex].months[monthIndex].documents[documentTypeIndex].documents, "name");

                        //Save Data to the Database
                        let DocumentsInTheFolder = directoryDetails.toJSON().documents[travesingType][yearIndex].months[monthIndex].documents[documentTypeIndex].documents[subDocumentTypeIndex].value;
                        DocumentsInTheFolder.push(...documentIds);
                        directoryDetails.documents[travesingType][yearIndex].months[monthIndex].documents[documentTypeIndex].documents[subDocumentTypeIndex].value = DocumentsInTheFolder;
                    } else {
                        //Save Data to the Database
                        let DocumentsInTheFolder = directoryDetails.toJSON().documents[travesingType][yearIndex].months[monthIndex].documents[documentTypeIndex].value;
                        DocumentsInTheFolder.push(...documentIds);
                        directoryDetails.documents[travesingType][yearIndex].months[monthIndex].documents[documentTypeIndex].value = DocumentsInTheFolder;
                    }

                } else {

                    // Only Document Types Needed For Non Recurring
                    const documentTypeIndex = findIndexOnObject(documentType, directoryDetails.toJSON().documents[travesingType], "name");

                    let DocumentsInTheFolder = directoryDetails.toJSON().documents[travesingType][documentTypeIndex].value;
                    DocumentsInTheFolder.push(...documentIds);
                    directoryDetails.documents[travesingType][documentTypeIndex].value = DocumentsInTheFolder;

                }

                directoryDetails.save().then((directoryDetail) => {
                    resolve(directoryDetail);
                }).catch((e) => {
                    // console.log(e);
                    reject(false);
                });
            }).catch((e) => {
                // console.log(e);
                reject(false);
            });
        } catch (e) {
            // console.log(e);
            reject(false);
        }
    });
}

const ValidateDocumentTypes = (type, isRecurring) => {
    const recurringFolders = ["cfdis", "declaración retención isr", "pago retención isr", "retención iva", "pago iva", "sua imss", "pago imss", "opinión imss", "opinión infonavit", "opinión sat", "resumen de liquidación sua"];
    const nonRecurringFolders = ["autorización stps", "contrato", "documentación corporativa"];
    if (isRecurring.toLowerCase() == "true") {
        return (recurringFolders.includes(type.toLowerCase()))
    } else {
        return (nonRecurringFolders.includes(type.toLowerCase()))
    }
};


const findIndexOnObject = (value, ObjectArray, key) => {
    try {
        let location = null;
        ObjectArray.map((singleItem, index) => {
            if (singleItem[key] == value) {
                location = index
            }
        });
        return location;
    } catch (e) {
        return null;
    }
}

const checkIfDirectoryBelongsToUser = (directoryId, userId) => {
    return new Promise(async (resolve, reject) => {
        try {
            const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
            const user = await User.findOne({ _id: objectId(userId) }).lean();
            if (DirectoryAccountMap) {
                if ((DirectoryAccountMap.clientId == userId || DirectoryAccountMap.providerId == userId || (DirectoryAccountMap.clientId.equals(user.clientId) && user.roles == 'subclient')) && DirectoryAccountMap.status == 'active') {
                    resolve(true);
                } else {
                    resolve(false);
                }
            } else {
                reject(false);
            }
        } catch (err) {
            reject(false);
        }
    });
}

const cleanUpDocumentData = (documentObjectArray) => {
    documentObjectArray.map((document) => {
        delete document["_id"];
        delete document["s3Path"];
        if (document["DocumentType"] == "CFDIs") {
            // Format JSON Data.
            document["data"] = document["DocumentData"] ? CFDIJSONFormatHelper(document["DocumentData"]) : {};
        } else if (document["DocumentType"] == "Declaración Retención ISR") {
            document["data"] = document["DocumentData"] ? declaracionRetencionISTFormatHelper(document["DocumentData"]) : {};
        }
        delete document["DocumentData"];
    });
    return documentObjectArray;
}

const ParseXMLFromDocument = async (documentObjectArray, pdfDocumentObjectArray) => {
    const response = documentObjectArray.map((document) => {
        if (document["DocumentType"] == "CFDIs" && document.isValid) {
            if (document["DocumentData"]) {
                return CFDIJSONFormatHelper(document["DocumentData"])
            }
        }
    });
    const compactedArray = array.compact(response);
    if (pdfDocumentObjectArray.length > 0) {
        // Checking CURP existence in PDF document and include status SI/NO
        const checkedCurpResponse = await checkCurpInPdfDocument(compactedArray, pdfDocumentObjectArray);
        return checkedCurpResponse;
    } else return compactedArray;
}

const checkCurpInPdfDocument = (dataArray, pdfDocumentObjectArray) => {
    // Combine PDFs pages text in a single variable
    const pdfDocumentObjectArrayRes = pdfDocumentObjectArray.map(async (documentObj) => {
        const data = JSON.parse(documentObj["DocumentData"]);
        var mergedData = mergeValues(data);
        var key = mergedData["Delegación IMSS"] + ' / ' + mergedData["SubDelegación IMSS"];
        var combinedText = await combinePdfAllPagesText(mergedData, data.length);
        return { name: key, combinedText: combinedText }
    });

    return Promise.all(pdfDocumentObjectArrayRes).then((data) => {
        const response = dataArray.map((row) => {
            // In number of PDFs CURP exist
            var numOfLocations = 0;
            data.map((documentObj, index) => {
                if (documentObj.combinedText.includes(row.CURP)) {
                    row[documentObj.name] = "SI";
                    numOfLocations++;
                } else row[documentObj.name] = "NO";

                if (index == (data.length - 1)) {
                    row["NÚMERO DE LOCACIONES ENCONTRADAS"] = numOfLocations;
                }
            });
            return row;
        });
        return Promise.resolve(response);
    });
}


const extractInformationFromDocument = (DocumentType, fileObject) => {
    return new Promise(async (resolve, reject) => {
        try {
            switch (DocumentType) {
                case "CFDIs":
                    const xml = fs.readFileSync(fileObject.path);
                    parseString(xml, function (err, result) {
                        if (err) {
                            reject(false);
                        }
                        const data = JSON.stringify(result);
                        resolve(data);
                    });
                    break;
                default:
                    const dataBuffer = fs.readFileSync(fileObject.path);
                    // Resolve Text Data
                    const options = { pagerender: pdf_render_function }
                    pdf(dataBuffer, options).then(function (data) {
                        data.text = data.text.replace(/(\r\n|\n|\r)/gm, "");
                        data.text = data.text.slice(0, -1);
                        data.text = "[" + data.text + "]";
                        resolve(data.text);
                    }).catch(function (error) {
                        console.log(error);
                        reject(false);
                    });
            }
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const CFDIJSONFormatHelper = (JSONString) => {
    try {
        const dataObject = JSON.parse(JSONString);
        let FechaTimpradoDate = new Date(dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["tfd:TimbreFiscalDigital"][0].$.FechaTimbrado);

        let FechaPagoDate = '', Deducciones = '', Curp = '', NumDiasPagados = '', RegistroPatronal = '';
        if (dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"]) {

            let nomina12 = dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"];

            FechaPagoDate = new Date(nomina12[0].$.FechaPago);
            Deducciones = nomina12[0]["nomina12:Deducciones"];
            Curp = nomina12[0]["nomina12:Receptor"][0].$.Curp;
            NumDiasPagados = nomina12[0].$.NumDiasPagados;
            RegistroPatronal = (typeof nomina12[0]["nomina12:Emisor"][0] === 'object') ? nomina12[0]["nomina12:Emisor"][0].$.RegistroPatronal : '';

        }
        let Descuento = (dataObject["cfdi:Comprobante"].$.Descuento) ? dataObject["cfdi:Comprobante"].$.Descuento : '';

        let curatedData = {
            UUID: dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["tfd:TimbreFiscalDigital"][0].$.UUID,
            TipoComprobante: dataObject["cfdi:Comprobante"].$.TipoDeComprobante,
            // Ano: new Date(dataObject["cfdi:Comprobante"].$.Fecha).getFullYear(),
            // Mes: new Date(dataObject["cfdi:Comprobante"].$.Fecha).getMonth(),
            // Dia: new Date(dataObject["cfdi:Comprobante"].$.Fecha).getDay(),
            NombreEmisor: dataObject["cfdi:Comprobante"]["cfdi:Emisor"][0].$.Nombre,
            RFCEmisor: dataObject["cfdi:Comprobante"]["cfdi:Emisor"][0].$.Rfc,
            NombreReceptor: dataObject["cfdi:Comprobante"]["cfdi:Receptor"][0].$.Nombre,
            RFCReceptor: dataObject["cfdi:Comprobante"]["cfdi:Receptor"][0].$.Rfc,
            RegistroPatronal: RegistroPatronal,
            CURP: Curp,
            Descripcion: dataObject["cfdi:Comprobante"]["cfdi:Conceptos"][0]["cfdi:Concepto"][0].$.Descripcion,
            NumDiasPagados: NumDiasPagados,
            FechaTimprado: (FechaTimpradoDate) ? (FechaTimpradoDate.getMonth() + 1) + '/' + FechaTimpradoDate.getDate() + '/' + FechaTimpradoDate.getFullYear() : '',
            FechaPago: (FechaPagoDate) ? (FechaPagoDate.getMonth() + 1) + '/' + FechaPagoDate.getDate() + '/' + FechaPagoDate.getFullYear() : '',
            SubTotal: dataObject["cfdi:Comprobante"].$.SubTotal,
            Descuento: Descuento,
            Total: dataObject["cfdi:Comprobante"].$.Total,
            IMSS: '',
            ISR: '',
            // FechaEmision: new Date(dataObject["cfdi:Comprobante"].$.Fecha).toDateString(),
            // FechaTimbrado: new Date(dataObject["cfdi:Comprobante"].$.Fecha).toTimeString(),
            // NoCertificadoEmisor : dataObject["cfdi:Comprobante"].$.NoCertificado,
            // NoCertificadoSAT: dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["tfd:TimbreFiscalDigital"][0].$.NoCertificadoSAT,
            // Version: dataObject["cfdi:Comprobante"].$.Version,
            // ResidenciaFiscal: ,
            // UsoCFDI: ,
            // TotalIEPS: ,
            // IVA16Percent: ,
            // RetenidoIVA: ,
            // RetenidoISR: ,
            // ISH: ,
            // RegimenFiscal: ,
            // LugarDeExpedicion: ,
            // FechaPago: ,
            // Serie: ,
            // Folio: ,
            // UUID: ,
            // TotalOriginal: ,
            // TotalTrasladados: ,
            // TotalRetenidos: ,
            // TotalLocalTrasladado: ,
            // TotalLocalRetenido: ,
            // Complemento: ,
            // Moneda: ,
            // TipoDeCambio: ,
            // FormaDePago: ,
            // MetododePago: ,
            // NumCtaPago: ,
            // CondiciondePago: ,
            // Conceptos: ,
        };

        if (Deducciones) {
            Deducciones[0]["nomina12:Deduccion"].map((Deduccion) => {
                if (Deduccion.$.Concepto == "IMSS") {
                    curatedData.IMSS = Deduccion.$.Importe;
                } else if (Deduccion.$.Concepto.includes("ISR")) {
                    curatedData.ISR = Deduccion.$.Importe;
                } else { }
            });
        }
        return curatedData;
    } catch (e) {
        console.log(e);
        return {};
    }
};

const declaracionRetencionISTFormatHelper = (DocumentString) => {
    try {
        let curatedData = {
            RFC: DocumentString.match(/(?<=RFC)[*\S]*/)[0] //Get the RFC Id
        }
        return curatedData
    } catch (e) {
        console.log(e);
        return {};
    }
}

let al_valor_agregado = false;
// PDF Render Page CallBack
function pdf_render_function(pageData) {
    let render_options = {
        normalizeWhitespace: true,
        disableCombineTextItems: true
    }

    return pageData.getTextContent(render_options)
        .then(function (textContent) {
            const pdfKeys = ["RFC", "EJERCICIO", "PERIOD", "A CARGO", "Ejercicio", "Período de la declaración", "Importe total a pagar", "PERIODO", "IMPUESTO CAUSADO", "Bimestre de Proceso", "SubDelegación IMSS", "Fecha de Pago", "Total de Pago", "Razón Social", "Periodo de pago"];
            const nameKeys = ["DECLARACIÓN PROVISIONAL O DEFINITIVA DE IMPUESTOS FEDERALES", "INFORMACIÓN REGISTRADA DE PAGOS DE CONTRIBUCIONES FEDERALES", "Opinión del cumplimiento de obligaciones fiscales", "Opinión del Cumplimiento de Obligaciones en materia de Seguridad Social", "Opinión del Cumplimiento de Obligaciones Fiscales en materia de Seguridad Social", "Asunto Constancia de Situación Fiscal.", "SISTEMA ÚNICO DE AUTODETERMINACIÓN", "FORMATO PARA PAGO DE CUOTAS OBRERO PATRONALES, APORTACIONES Y AMORTIZACIONES", "Declaración Provisional o Definitiva de Impuestos Federales"];
            // let lastY = '';
            let text = {};
            let cantidad_a_cargo = [];
            let items = textContent.items;
            let pdfAllText = '';
            // console.log(items.map((row) => row.str));
            for (let i = 0; i < items.length; i++) {
                let str = items[i].str;
                let prevStr = (i > 0) ? items[i - 1].str : "";
                let nextStr = (items[i + 1]) ? items[i + 1].str : "";

                str = str.replace(":", "");
                prevStr = prevStr.replace(":", "");
                nextStr = nextStr.replace(":", "");

                if (pdfKeys.includes(str)) {
                    if (str == "Período de la declaración") {
                        text[str] = (nextStr) ? nextStr : items[i + 2].str;
                    } else text[str] = nextStr;
                    i++;
                } else if (str == "DETERMINACIÓN DEL IMPUESTO AL VALOR AGREGADO") {
                    al_valor_agregado = true;
                } else if (str == "CANTIDAD A FAVOR" && al_valor_agregado) {
                    text[str] = (nextStr != "OTROS ESTÍMULOS") ? nextStr : 0;
                } else if (str == "CANTIDAD A PAGAR" && al_valor_agregado) {
                    text[str] = nextStr;
                    i++;
                } else if (str.includes('Concepto de pago')) {
                    let data = {};
                    data['Concepto de pago'] = nextStr;
                    let cantidad_a_pag = '';
                    i++;
                    for (let j = i; j < items.length; j++) {
                        let strChk = items[j].str.replaceAll(':', '');
                        if (strChk == 'Cantidad a pagar' && !cantidad_a_pag) {
                            cantidad_a_pag = items[j + 1].str;
                        }
                    }
                    data['Cantidad a pagar'] = cantidad_a_pag;
                    cantidad_a_cargo.push(data);
                } else if (nameKeys.includes(str)) {
                    text.name = str;
                } else if ((str == "ISR RETENCIONES POR SALARIOS" || str == "IMPUESTO AL VALOR AGREGADO") && nextStr == "Impuesto a cargo") {
                    text[str] = { "Impuesto a cargo": items[i + 2].str };
                    i += 2;
                } else if (str == "Importe total" && nextStr == "a pagar") {
                    text[str + " " + nextStr] = items[i + 2].str;
                    i += 2;
                } else if (str == "INFORMACIÓN DEL PAGO RECIBIDO EN LA INSTITUCIÓN DE CRÉDITO AUTORIZADA") {
                    text[str] = true;
                } else if (str == "Clave de R.F.C.") {
                    if (nextStr == "Folio") {
                        text[str] = items[i + 2].str;
                        i += 2;
                    } else if (nextStr == "Nombre, Denominación o Razón Social") {
                        text[str] = items[i + 5].str;
                        i += 5;
                    }
                } else if (str == "Clave R.F.C.") {
                    text['Clave de R.F.C.'] = items[i + 3].str;
                } else if (str.includes("POSITIVO")) {
                    text["POSITIVO"] = true;
                } else if ((str.includes("Revisión practicada el") || str.includes("Ciudad de")) && !text["dateText"]) {
                    text["dateText"] = str;
                } else if (str.includes("Esta carta opinión de cumplimiento de obligaciones tiene")) {
                    text["dateText"] = str;
                } else if (str.includes("se encuentra al corriente con las obligaciones de pago") || str.includes("se emite opinión Positiva") || str.includes("NO se identificaron adeudos ante el Infonavit")) {
                    text["opinionIsPositive"] = true;
                } else if (str == "El Instituto del Fondo Nacional de la Vivienda para los " && nextStr.includes("Trabajadores")) {
                    text["issuedFrom"] = str + "Trabajadores";
                } else { }

                if (str == "CANTIDAD A PAGAR" && items[i - 2].str == "CANTIDAD A CARGO") {
                    text[str] = nextStr;
                }

                if (text.name && text.name == "SISTEMA ÚNICO DE AUTODETERMINACIÓN" && nextStr == "Delegación IMSS" && str == "RFC") {
                    text[str] = items[i + 10].str.replaceAll("-", "").trim();
                    i += 10;
                }

                if (text.name && text.name == "SISTEMA ÚNICO DE AUTODETERMINACIÓN" && nextStr == "Actividad " && str == "RFC") {
                    if (items[i + 1].str == "Area Geográfica:") {
                        text[str] = items[i + 7].str.replaceAll("-", "").trim();
                        i += 7;
                    } else {
                        text[str] = items[i + 6].str.replaceAll("-", "").trim();
                        i += 6;
                    }
                }

                if (text.name && text.name == "SISTEMA ÚNICO DE AUTODETERMINACIÓN" && str.includes("Período de Proceso")) {
                    text['Período de Proceso'] = str.replace("Período de Proceso ", "");
                }

                if (str.includes("STPS")) text.STPS = true;
                if (str === "CUMPLE") text.CUMPLE = true;
                if (str.replaceAll(' ', '').includes('vigenciadeTRESAÑOS')) text.tresAnos = true;

                if (str.includes("Federal de Contribuyentes") || str.includes("Contribuyentes")) {
                    let strSplit = nextStr.split(',');
                    text.RFC = strSplit[0].replaceAll(" ", "");
                }

                if (str.includes("NRP")) {
                    let strSplit = str.split('NRP');
                    text['NRP'] = strSplit[1].replaceAll(" ", "");
                }

                if (str == "Delegación IMSS") {
                    text[str] = prevStr;
                }

                if (text.name == "FORMATO PARA PAGO DE CUOTAS OBRERO PATRONALES, APORTACIONES Y AMORTIZACIONES") {
                    if (str.toLowerCase() == "registro patronal") {
                        if (prevStr == 'LC-2') {
                            text['Registro Patronal'] = items[i - 15].str;
                            text['razon_social'] = items[i - 16].str;
                        } else {
                            text['Registro Patronal'] = prevStr;
                            text['razon_social'] = items[i - 3].str;
                        }
                    }

                    if (str.includes("PERÍODO QUE COMPRENDE")) {
                        if (items[i + 4].str != 'LC-2') text['dateText'] = items[i + 3].str;
                        else text['dateText'] = items[i + 7].str;
                    }

                    if (str.replaceAll(' ', '') == "TOTALAPAGAR") text['total_a_pagar'] = items[i + 88].str;
                } else {
                    if (str.replaceAll(' ', '') == "TOTALAPAGAR") text['total_a_pagar'] = nextStr.replaceAll(' ', '');
                    if (str.includes('Mes de Proceso')) text['dateText'] = nextStr;

                    if (str == "Registro Patronal") {
                        text['Registro Patronal'] = items[i + 5].str;
                    }

                    if (str == "Nombre o Razón Social") {
                        text['razon_social'] = items[i - 7].str;
                    }
                }

                if (((prevStr + str + nextStr).replaceAll(" ", "")).includes('NOseidentificaronadeudosanteelInfonavit')) text["opinionIsPositive"] = true;

                // For Pago IMSS multiple type files
                if (str.includes('FECHA DE CARGO')) {
                    if (str.includes('FECHA DE CARGO') && nextStr == 'HORA DE CARGO') {
                        text['Fecha de Pago'] = items[i + 2].str;
                    } else {
                        text['Fecha de Pago'] = nextStr;
                    }
                }

                if (str.includes('NOMBRE O RAZON SOCIAL') && nextStr == 'REGISTRO PATRONAL') {
                    text['Razón Social'] = items[i + 4].str;
                } else if (str.includes('NOMBRE O RAZON SOCIAL')) {
                    text['Razón Social'] = '';
                    ++i;
                    while (items[i].str != 'REGISTRO PATRONAL') {
                        text['Razón Social'] += items[i].str;
                        ++i;
                    }
                } else { }

                if (str == "RFC" && nextStr == "FOLIO SUA") text["RFC"] = (items[i + 3].str).trim();
                if (str == "PERIODO IMSS" && nextStr == "VERSION SUA") text['Periodo de pago'] = items[i + 3].str;


                if (str.includes('TOTAL A PAGAR')) text['Total de Pago'] = items[i + 9].str;
                if (str.includes('TOTAL A PAGAR') && items[i + 4].str == "BBVA ENTIDAD RECEPTORA") text['Total de Pago'] = items[i + 3].str;

                if (prevStr == 'C' && str == 'l' && nextStr == 'a' && items[i + 2].str == 'v' && items[i + 3].str == 'e') {
                    if ((items[i + 13].str).length == 1) {
                        i += 13;
                        while (items[i].str != 'Nombre, Denominación o Razón social') {
                            text['Clave de R.F.C.'] += items[i].str;
                            ++i;
                        }
                        text['Clave de R.F.C.'] = (text['Clave de R.F.C.'].replace('undefined', '')).trim();
                    } else text['Clave de R.F.C.'] = items[i + 13].str;
                }

                // For Pago IMSS, Santander Bank
                if (str == "Cuenta de" && nextStr == "Cargo") text['Razón Social'] = (items[i + 2].str).trim();
                if (str == "Período de" && nextStr == "Pago") text['Periodo de Pago'] = (items[i + 2].str).trim();
                if (str == 'Importe Total' && nextStr != "a pagar") text['Total de Pago'] = (nextStr.replace(/\b(?:MXN|MN)\b/gi, '')).trim();
                // For Santander Bank new format
                if (str == "NOMBRE O RAZÓN SOCIAL") text['Razón Social'] = nextStr;
                if (str == "AÑO" && prevStr == "MES") {
                    text['mes'] = items[items.length - 3].str;
                    text['ano'] = items[items.length - 4].str;
                }
                if (str == "RFC" && nextStr == "FOLIO/SUA") text["RFC"] = items[i + 68].str
                if (str == "TOTAL A" && nextStr == "PAGAR") text['Total de Pago'] = items[items.length - 8].str;
                //  For BanBajio Bank
                if (str == "Capturó") text['Razón Social'] = nextStr.trim();
                // For INBURSA Bank
                if (str == "Total a pagar" || str == "Total  a Pagar") {
                    if (items[i + 2]?.str == "Cuota Fija") {
                        text['Total de Pago'] = prevStr.trim();
                    } else {
                        text['Total de Pago'] = nextStr.trim();
                    }
                }

                if (str == "DATOS DEL PATRÓN" && nextStr == "Registro patronal") text['Razón Social'] = (items[i + 2].str).trim();
                // For Banorte Bank
                if (str == "NOMBRE O RAZÓN SOCIAL" && nextStr == ' ') {
                    text['Razón Social'] = '';
                    ++i;
                    while (items[i].str != 'DOMICILIO:') {
                        text['Razón Social'] += items[i].str;
                        ++i;
                    }
                }
                if (str == "IMSS" && nextStr == "MES" && items[i + 4].str == "AÑO") {
                    text['mes'] = items[i + 3].str;
                    text['ano'] = items[i + 6].str;
                }
                if (str == "TOTAL A PAGAR" && items[i - 2].str == "SUB-TOTAL") {
                    text['Total de Pago'] = nextStr;
                }
            }

            if (text.name == "SISTEMA ÚNICO DE AUTODETERMINACIÓN") {
                for (let item of textContent.items) {
                    pdfAllText += item.str + " ";
                }
                text['page' + items[items.length - 2].str] = pdfAllText;
            }

            if (text.name == "DECLARACIÓN PROVISIONAL O DEFINITIVA DE IMPUESTOS FEDERALES") {
                text.cantidad_a_cargo = cantidad_a_cargo;
            }

            // for (let item of textContent.items) {
            //     if (pdfKeys.includes(item.str)) {
            //         if (lastY == item.transform[5] || !lastY){
            //             text += item.str + " ";
            //         }
            //         else{
            //             // text += '\n' + item.str;
            //             text += ',' + item.str;
            //         }
            //         lastY = item.transform[5];
            //     }
            // }
            // console.log(text);
            text = JSON.stringify(text) + ",";
            return text;
        });
}


const checkDisperancy = (type, dataArray) => {
    switch (type) {
        case 'year':
            // #1 Get the current year.
            const currentYear = new Date().getFullYear();
            // #2  Check if current year exists in the data, If it does then we don't have to do anything
            if (dataArray.includes(`${currentYear}`)) {
                return false;
            }

            // #3 if it doesn't exist, Then we need to return the number of new year objects we need to add to the directory.
            // Formula for that would be, the highest value in existing array - the current year

            let yearsArray = [];
            let highestYear = Math.max(...dataArray);
            while (highestYear < currentYear) {
                yearsArray.push(++highestYear);
            };
            return yearsArray;
        case 'month':
            // #1 Get current Month
            const currentMonth = new Date().getMonth() + 1

            // #2  Check if current year exists in the data, If it does then we don't have to do anything
            if (dataArray.includes(`${currentMonth}`)) {
                return false;
            }

            // #3 if it doesn't exist, Then we need to return the number of new year objects we need to add to the directory.
            // Formula for that would be, the highest value in existing array - the current year

            let monthsArray = [];
            let highestMonth = Math.max(...dataArray);
            while (highestMonth < currentMonth) {
                monthsArray.push(++highestMonth);
            };
            return monthsArray;
        default:
            return false;
    }
}

const addYearsToDirectory = (directoryId, yearsArray) => {
    return new Promise((resolve, reject) => {
        try {
            Directory.findById(objectId(directoryId)).then((directoryData) => {
                // If no response for the directory Id, Return False promise
                if (!directoryData) {
                    reject(false)
                }
                // Map years into the directory, Create New Templates, Month will always be 1
                yearsArray.map((year) => {
                    directoryData.documents.reccuring.push(
                        {
                            year: year,
                            months: [
                                {
                                    month: 1,
                                    documents: monthlyDocumentsTemplate
                                }
                            ]
                        }
                    );
                });

                // Save the Directory and send a positive promise & also return only new data for years

                directoryData.save().then((directory) => {
                    if (!directory) {
                        reject(false);
                    }
                    console.log("Directory Modified : ", directory._id.toString())

                    let documents = directory.toJSON().documents.reccuring;
                    yearsData = documents.map((documentData) => {
                        delete documentData.months;
                        delete documentData._id;
                        return documentData.year;
                    });
                    resolve(yearsData);
                }).catch((err) => {
                    // Console the Error
                    console.log(err);
                    reject(false)
                });

            }).catch((err) => {
                // Console the Error
                console.log(err);
                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}

const addMonthsToDirectory = (directoryId, year, MonthsArray) => {
    return new Promise((resolve, reject) => {
        try {
            Directory.findById(objectId(directoryId)).then((directoryData) => {
                // If no response for the directory Id, Return False promise
                if (!directoryData) {
                    reject(false)
                }
                // Map months into the directory, Create New Templates
                directoryData.documents.reccuring.map((eachYear, index) => {
                    if (eachYear.year == year) {
                        MonthsArray.map((month) => {
                            directoryData.documents.reccuring[index].months.push({
                                month: month,
                                documents: monthlyDocumentsTemplate
                            })
                        });
                    }
                })

                // Save the Directory and send a positive promise & also return only new data for months
                directoryData.save().then((directory) => {
                    if (!directory) {
                        reject(false);
                    }
                    console.log("Directory Modified : ", directory._id.toString())

                    const documents = directory.toJSON().documents.reccuring;
                    let monthsData = [];
                    documents.forEach((documentData) => {
                        if (documentData.year == year) {
                            documentData.months.map((monthsObject) => {
                                monthsData.push(monthsObject.month)
                            });
                        }
                    });
                    resolve(monthsData);
                }).catch((err) => {
                    // Console the Error
                    console.log(err);
                    reject(false)
                });
            }).catch((err) => {
                // Console the Error
                console.log(err);
                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}

const CountTotalAccount = async (id) => {
    var count = 0;
    const DirectoryAccountMap = await model('accountsRelations').find({ clientId: objectId(id) }).lean();
    // console.log(DirectoryAccountMap);
    for (const t2 of DirectoryAccountMap) {
        count = count + 1;
    }
    return count;
}

const CountTotalUploadedFiles = async (id, sumtype) => {
    var count2 = 0;
    var TotalRecurringDocumment = 0;
    var TotalNonRecurringDocumment = 0;
    var TotalDocuments = 0;
    var currentYear = new Date().getFullYear();
    var Currentmonth = new Date().getMonth() + 1;
    if (sumtype == "last_month") {
        if (Currentmonth != 1) {
            Currentmonth = Currentmonth - 1;
        } else {
            Currentmonth = 12;
            currentYear = currentYear - 1;
        }
    }
    const AccountRelationData = await model('accountsRelations').find({ providerId: objectId(id) });
    count2 = AccountRelationData.length;
    if (count2 > 0) {
        const DocumentDirectoryID = AccountRelationData[0].directory;
        const DirectoryData = await Directory.find({ _id: DocumentDirectoryID });
        DataCount = DirectoryData.length;
        if (DataCount > 0) {
            // Recurring calculation
            var recurringArray = DirectoryData[0].documents.reccuring;
            for (var key in recurringArray) {
                var obj = recurringArray[key];
                if (obj.year == currentYear) {
                    const MonthsArray = obj.months;
                    for (var monthkey in MonthsArray) {
                        var Monthobj = MonthsArray[monthkey];
                        if (Monthobj.month == Currentmonth) {
                            const DocumentArray = Monthobj.documents;
                            for (var documentkey in DocumentArray) {
                                var Documentobj = DocumentArray[documentkey];
                                var documentCount = Documentobj.value.length;
                                TotalRecurringDocumment = TotalRecurringDocumment + documentCount;
                            }
                        }
                    }
                }
            }
            //No Recurring calculation
            var NorecurringArray = DirectoryData[0].documents.nonRecurring;
            for (var norecurringdocumentkey in NorecurringArray) {
                var NoRecuDocumentobj = NorecurringArray[norecurringdocumentkey];
                var NoRecudocumentCount = NoRecuDocumentobj.value.length;
                TotalNonRecurringDocumment = TotalNonRecurringDocumment + NoRecudocumentCount;
            }

        }

    }
    TotalDocuments = TotalDocuments + TotalRecurringDocumment;
    return TotalDocuments;

}
const parsePDFDataFromDocument = (directoryId, user, year, month, documents) => {
    return new Promise((resolve, reject) => {
        try {
            let response = { "CFDIs": [], "SUA IMSS": [] };

            // Getting data from non recurring folder
            documents.nonRecurring.forEach(async (documentData, index) => {
                if (documentData.name === "Autorización STPS") {
                    nonRecurringPromises = documentData.value.map(async (documentId) => {
                        const documentObj = await Document.findOne({ _id: documentId }).lean();
                        if (documentObj && documentObj.isValid && documentObj["DocumentData"]) {
                            // console.log(documentObj["DocumentData"]);
                            const data = JSON.parse(documentObj["DocumentData"]);
                            response[documentData.name] = data;
                        }
                    });
                }

                if (index == documents.nonRecurring.length - 1) {
                    const execNonRecurringPromise = await Promise.all(nonRecurringPromises);
                }

            });

            // Getting data from recurring folder
            documents.reccuring.forEach((documentData) => {
                if (documentData.year == year) {
                    documentData.months.map(async (monthsObject) => {
                        if (monthsObject.month == month) {
                            promises = monthsObject.documents.map(async (document, index) => {
                                if (document.name == "Resumen de Liquidación SUA") {
                                    parseAllDoc = document.documents.map(async (subDocuments) => {
                                        parseSubDoc = subDocuments.value.map(async (documentId) => {
                                            const documentObj = await Document.findOne({ _id: documentId, isValid: true }).lean();
                                            if (documentObj) {
                                                if (documentObj["DocumentData"]) {
                                                    const data = JSON.parse(documentObj["DocumentData"]);
                                                    response[subDocuments.name] = data;
                                                }
                                            }
                                        });
                                        await Promise.all(parseSubDoc);
                                    });

                                    // Wait till all document data processed
                                    const execParseAllDoc = await Promise.all(parseAllDoc);
                                } else {
                                    parseAllDoc = document.value.map(async (documentId) => {
                                        const documentObj = await Document.findOne({ _id: documentId, isValid: true }).lean();
                                        if (documentObj) {
                                            if (documentObj["DocumentType"] == "CFDIs") {
                                                if (documentObj["DocumentData"]) {
                                                    const data = CFDIJSONFormatHelper(documentObj["DocumentData"]);
                                                    response[document.name].push(data);
                                                }
                                            } else {
                                                if (documentObj["DocumentData"]) {
                                                    const data = JSON.parse(documentObj["DocumentData"]);
                                                    if (documentObj["DocumentType"] == "SUA IMSS") response[document.name].push(data);
                                                    else response[document.name] = data;
                                                }
                                            }
                                        }
                                    });
                                    // Wait till all document data processed
                                    const execParseAllDoc = await Promise.all(parseAllDoc);
                                }
                            });

                            // Wait till all months' documents processed
                            Promise.all(promises).then(async () => {
                                const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
                                const Provider = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();
                                const Client = await User.findOne({ _id: objectId(DirectoryAccountMap.clientId) }).lean();
                                // console.log(response);
                                const result = await generatePdfData(Client, Provider, year, month, response);
                                resolve(result);
                            });
                        }
                    });
                }
            });
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const meses = [
    {
        "id": 1,
        "nombre": "Enero"
    },
    {
        "id": 2,
        "nombre": "Febrero"
    },
    {
        "id": 3,
        "nombre": "Marzo"
    },
    {
        "id": 4,
        "nombre": "Abril"
    },
    {
        "id": 5,
        "nombre": "Mayo"
    },
    {
        "id": 6,
        "nombre": "Junio"
    },
    {
        "id": 7,
        "nombre": "Julio"
    },
    {
        "id": 8,
        "nombre": "Agosto"
    },
    {
        "id": 9,
        "nombre": "Septiembre"
    },
    {
        "id": 10,
        "nombre": "Octubre"
    },
    {
        "id": 11,
        "nombre": "Noviembre"
    },
    {
        "id": 12,
        "nombre": "Diciembre"
    }
];

const generatePdfData = (client, provider, year, month, data) => {
    let monthName = meses.find(_mes => _mes.id == month).nombre;
    let monthNameLower = monthName.toLowerCase();
    return new Promise(async (resolve, reject) => {
        let result = {};
        result.Datos_generales = { client: provider.name, RFC: provider.rfc };
        try {
            const CFDIsTotal = calculateCFDIsTotal(data.CFDIs);
            result.CFDIs = CFDIsTotal;
            result.CFDIs.isValid = (data.CFDIs.length) ? true : false;

            if (data['Autorización STPS']) {
                let AutorizacionSTPS = data['Autorización STPS'][0];
                if (AutorizacionSTPS.STPS) {
                    const dateTextArr = AutorizacionSTPS.dateText.split(' ');
                    if (dateTextArr.length == 10) {
                        let day = dateTextArr[5];
                        let month = meses.find(_mes => _mes.nombre == dateTextArr[7]).id;
                        let year = dateTextArr[9];
                        AutorizacionSTPS.date = `${month}/${day}/${year}`;

                        // Checking for Autorización STPS PDF validity
                        const currentDate = new Date();
                        const AutorizacionSTPSDate = new Date(`${month}/${day}/${parseInt(year) + 3}`);

                        result.AutorizacionSTPS = AutorizacionSTPS;
                        result.AutorizacionSTPS.isValid = false;
                        if (AutorizacionSTPSDate.getTime() > currentDate.getTime() && AutorizacionSTPS.RFC == provider.rfc) {
                            result.AutorizacionSTPS.isValid = true;
                        }
                    }
                }
            }

            if (data['Declaración Retención ISR']) {
                let DeclaracionRetencionISR = data['Declaración Retención ISR'][0];
                result.DeclaracionRetencionISR = {
                    RFC: DeclaracionRetencionISR.RFC,
                    EJERCICIO: DeclaracionRetencionISR.Ejercicio,
                    PERIODO: DeclaracionRetencionISR['Período de la declaración']
                };
                result.DeclaracionRetencionISR.isValid = false;

                // if(parseFloat(result.DeclaracionRetencionISR['A CARGO']) > CFDIsTotal.ISRSum){
                //     result.DeclaracionRetencionISR.isValid = true;
                //     result.DeclaracionRetencionISR.Comentario = 'Cantidad a pagar mayor a ISR de CFDIS cargados';
                // } else {
                //     result.DeclaracionRetencionISR.isValid = false;
                //     result.DeclaracionRetencionISR.Comentario = 'Cantidad a pagar menor a ISR de CFDIS cargados';
                // }

            }

            if (data['Pago Retención ISR']) {
                let PagoRetencionISR = mergeValues(data['Pago Retención ISR']);
                let cargoSum = 0;
                result.PagoRetencionISR = {
                    RFC: PagoRetencionISR.RFC,
                    Ejercicio: PagoRetencionISR.Ejercicio,
                    Periodo: PagoRetencionISR['Período de la declaración'],
                    cantidad_a_cargo: PagoRetencionISR.cantidad_a_cargo
                };
                result.PagoRetencionISR.isValid = true;

                PagoRetencionISR.cantidad_a_cargo.forEach(row => cargoSum += parseFloat(row['Cantidad a pagar'].replaceAll(',', '')));

                if (result.DeclaracionRetencionISR) {
                    result.DeclaracionRetencionISR.cargo = cargoSum;
                    result.DeclaracionRetencionISR.isValid = true;
                }

                // if(result.PagoRetencionISR['ISR RETENCIONES POR SALARIOS'] && result.PagoRetencionISR['ISR RETENCIONES POR SALARIOS']['Impuesto a cargo']) {
                //     result.PagoRetencionISR.ISR = result.PagoRetencionISR['ISR RETENCIONES POR SALARIOS']['Impuesto a cargo'];
                // }
            }

            if (data['Retención IVA']) {
                let RetencionIVA = mergeValues(data['Retención IVA']);

                if (RetencionIVA['Período de la declaración']) RetencionIVA.PERIODO = RetencionIVA['Período de la declaración'];
                if (RetencionIVA.Ejercicio) RetencionIVA.EJERCICIO = RetencionIVA.Ejercicio;

                result.RetencionIVA = {
                    RFC: RetencionIVA.RFC,
                    EJERCICIO: RetencionIVA.EJERCICIO,
                    PERIODO: RetencionIVA.PERIODO,
                    cantidad_a_fav: (RetencionIVA['CANTIDAD A FAVOR']) ? RetencionIVA['CANTIDAD A FAVOR'] : ' ',
                    cantidad_a_pag: RetencionIVA['CANTIDAD A PAGAR']
                };
                result.RetencionIVA.isValid = true;

                // if(parseFloat(result.RetencionIVA['IMPUESTO CAUSADO']) > 0){
                //     result.RetencionIVA.isValid = true;
                //     result.RetencionIVA.Comentario = 'Impuesto causado mayor a 0';
                // } else {
                //     result.RetencionIVA.isValid = false;
                //     result.RetencionIVA.Comentario = 'Impuesto causado menor a 0';
                // }

                // if (RetencionIVA.RFC == provider.rfc) result.RetencionIVA.isValid = true;
            }

            if (data['Pago IVA']) {
                let PagoIVA = mergeValues(data['Pago IVA']);

                result.PagoIVA = {
                    RFC: PagoIVA.RFC,
                    Ejercicio: PagoIVA.Ejercicio,
                    Periodo: PagoIVA['Período de la declaración'],
                    cantidad_a_cargo: PagoIVA.cantidad_a_cargo,
                    isValid: true
                };

                // if(result.PagoIVA['IMPUESTO AL VALOR AGREGADO'] && result.PagoIVA['IMPUESTO AL VALOR AGREGADO']['Impuesto a cargo']) {
                //     result.PagoIVA.IVA = result.PagoIVA['IMPUESTO AL VALOR AGREGADO']['Impuesto a cargo'];
                // }

                // if (result.PagoRetencionISR && result.PagoRetencionISR.ISR) result.PagoIVA.ISR = result.PagoRetencionISR.ISR;

                // let DeclaracionRetencionISR_ACargo = ''; RetencionIVA_ACargo = '';

                // if (result.DeclaracionRetencionISR && result.RetencionIVA && ( result.PagoIVA.ISR && result.PagoRetencionISR.isValid ) && (PagoIVA.RFC == provider.rfc)) {
                //     DeclaracionRetencionISR_ACargo = parseInt(result.DeclaracionRetencionISR['A CARGO'].replace(/\$|,/g, ''));
                //     RetencionIVA_ACargo = parseInt(result.RetencionIVA['A CARGO'].replace(/\$|,/g, ''));

                //     result.PagoIVA['Importe total a pagar'] = result.PagoIVA['Importe total a pagar'].replace(/\$|,/g, '');

                //     let Importe_total_a_pagar = parseInt(result.PagoIVA['Importe total a pagar']);

                //     let bothCargoTotal = DeclaracionRetencionISR_ACargo + RetencionIVA_ACargo;

                //     if (result.PagoIVA.ISR == result.DeclaracionRetencionISR['A CARGO'] && bothCargoTotal <= Importe_total_a_pagar && (result.PagoIVA.IVA && result.PagoIVA.IVA == result.RetencionIVA['A CARGO'])) {
                //         result.PagoIVA.isValid = true;
                //     }
                // } else {
                //     result.PagoIVA.isValid = false;
                // }
            }

            if (data['SUA IMSS'].length > 0) {
                let SUA_IMSS = mergeValues(data['SUA IMSS'][0]);

                if (SUA_IMSS["Período de Proceso"]) {
                    let dateArr = SUA_IMSS["Período de Proceso"].split('-');
                    SUA_IMSS.Periodo = dateArr[0];
                    SUA_IMSS.Ejercicio = dateArr[1];
                } else if (SUA_IMSS["Bimestre de Proceso"]) {
                    let dateArr = SUA_IMSS["Bimestre de Proceso"].split('-');
                    SUA_IMSS.Periodo = dateArr[0];
                    SUA_IMSS.Ejercicio = dateArr[1];
                } else {
                    SUA_IMSS.Periodo = " ";
                    SUA_IMSS.Ejercicio = " ";
                }

                result.SUA_IMSS = {};
                result.SUA_IMSS.RFC = SUA_IMSS.RFC;
                result.SUA_IMSS.Periodo = SUA_IMSS.Periodo;
                result.SUA_IMSS.Ejercicio = SUA_IMSS.Ejercicio;
                result.SUA_IMSS.isValid = false;
                // Combine all PDF pages data in single text to check CURP
                let PDFsCombinedTextArray = await extractAllPDFsPagesText(data['SUA IMSS']);
                // All PDF locations
                result.SUA_IMSS.locations = PDFsCombinedTextArray.map((textObj) => textObj.name);
                // Checking all CURPS include in PDF
                let numOfCurpsFound = 0;
                let checkForCurps = data.CFDIs.map((row) => {
                    let check = PDFsCombinedTextArray.some((TextObj) => {
                        return TextObj.combinedText.includes(row.CURP);
                    });

                    if (check) numOfCurpsFound++;
                });
                // Wait till loop complete
                const resolvePromise = await Promise.all(checkForCurps);
                CheckAllCurpsInclude = (numOfCurpsFound == data.CFDIs.length);
                result.SUA_IMSS.Comentario = `${numOfCurpsFound}/${data.CFDIs.length} TRABAJADORES ENCONTRADOS`;

                if (result.SUA_IMSS.RFC == provider.rfc && CheckAllCurpsInclude) result.SUA_IMSS.isValid = true;
            }

            if (data['Opinión SAT']) {
                let OpinionSAT = mergeValues(data['Opinión SAT']);

                result.OpinionSAT = OpinionSAT;
                result.OpinionSAT.isValid = false;

                result.OpinionSAT.Estatus = (result.OpinionSAT.POSITIVO) ? 'Positiva' : 'Negativa';
                result.OpinionSAT.Periodo = (result.OpinionSAT.dateText.includes(monthName) || result.OpinionSAT.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionSAT.Ejercicio = result.OpinionSAT.dateText.includes(year) ? year : " ";

                if (OpinionSAT["Clave de R.F.C."] == provider.rfc && result.OpinionSAT.POSITIVO) result.OpinionSAT.isValid = true;
            }

            if (data['Opinión IMSS']) {
                let OpinionIMSS = mergeValues(data['Opinión IMSS']);

                result.OpinionIMSS = OpinionIMSS;
                result.OpinionIMSS.isValid = false;

                result.OpinionIMSS.Estatus = (result.OpinionIMSS.opinionIsPositive) ? 'Positiva' : 'Negativa';
                result.OpinionIMSS.Periodo = (result.OpinionIMSS.dateText.includes(monthName) || result.OpinionIMSS.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionIMSS.Ejercicio = result.OpinionIMSS.dateText.includes(year) ? year : " ";

                if (OpinionIMSS["Clave de R.F.C."] == provider.rfc && result.OpinionIMSS.opinionIsPositive)
                    result.OpinionIMSS.isValid = true;
            }

            if (data['Opinión INFONAVIT']) {
                let OpinionINFONAVIT = mergeValues(data['Opinión INFONAVIT']);

                result.OpinionINFONAVIT = OpinionINFONAVIT;
                result.OpinionINFONAVIT.isValid = false;

                result.OpinionINFONAVIT.Estatus = (result.OpinionINFONAVIT.opinionIsPositive) ? 'Positiva' : 'Negativa';
                result.OpinionINFONAVIT.Periodo = (result.OpinionINFONAVIT.dateText.includes(monthName) || result.OpinionINFONAVIT.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionINFONAVIT.Ejercicio = result.OpinionINFONAVIT.dateText.includes(year) ? year : " ";

                if (result.OpinionINFONAVIT.opinionIsPositive) result.OpinionINFONAVIT.isValid = true;
            }

            if (data['Resumen de Liquidación SUA 1st vers'] || data['Resumen de Liquidación SUA 2nd vers']) {
                resumendeLiquidacionSUA = {
                    isValid: false,
                };
                let firstTotal = 0, secondTotal = 0;

                if (data['Resumen de Liquidación SUA 1st vers']) {
                    let firstData = data['Resumen de Liquidación SUA 1st vers'][0];
                    resumendeLiquidacionSUA['razon_social'] = firstData.razon_social;
                    resumendeLiquidacionSUA['registro_patronal'] = firstData['Registro Patronal'];
                    firstTotal = parseFloat(firstData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (data['Resumen de Liquidación SUA 2nd vers']) {
                    let secondData = data['Resumen de Liquidación SUA 2nd vers'][0];
                    resumendeLiquidacionSUA['razon_social'] = secondData.razon_social;
                    resumendeLiquidacionSUA['registro_patronal'] = secondData['Registro Patronal'];
                    secondTotal = parseFloat(secondData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                resumendeLiquidacionSUA.total_a_pagar = (firstTotal >= secondTotal) ? firstTotal : secondTotal;

                if (firstTotal && secondTotal && (firstTotal == secondTotal)) resumendeLiquidacionSUA.isValid = true;

                result.resumendeLiquidacionSUA = resumendeLiquidacionSUA;
            }

            if (data['PAGO IMSS']) {
                let PagoImss = mergeValues(data['PAGO IMSS']);

                if (PagoImss['ano']) {
                    PagoImss['Fecha de Pago'] = (meses.find(_mes => _mes.id == parseInt(PagoImss['mes'])).nombre) + ' ' + PagoImss['ano'];
                }

                if (PagoImss['Periodo de pago']) {
                    PagoImss['Fecha de Pago'] = (meses.find(_mes => _mes.id == parseInt(PagoImss['Periodo de pago'].substring(4, 6))).nombre) + ' ' + parseInt(PagoImss['Periodo de pago'].substring(0, 4));
                }

                result.PagoImss = {
                    razon_social: PagoImss['Razón Social'],
                    fecha_de_pago: (PagoImss['Fecha de Pago']) ? PagoImss['Fecha de Pago'] : PagoImss['Periodo de Pago'],
                    total_de_pago: (PagoImss['Total de Pago']) ? PagoImss['Total de Pago'].replace(/[$,]/g, '') : '',
                    isValid: false
                };

                if (result.resumendeLiquidacionSUA && result.resumendeLiquidacionSUA.total_a_pagar == result.PagoImss.total_de_pago) result.PagoImss.isValid = true;
            }

            resolve(result);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const mergeValues = (values) => {
    result = {};
    cantidad_a_cargo = [];
    values.map((value) => {
        Object.keys(value).map((key) => {
            if (!result[key] && key != 'cantidad_a_cargo') {
                result[key] = value[key];
            } else if (key == "cantidad_a_cargo") {
                cantidad_a_cargo = [...cantidad_a_cargo, ...value[key]];
            } else { }
        });
    });

    if (cantidad_a_cargo.length) result.cantidad_a_cargo = cantidad_a_cargo;

    return result;
}

const calculateCFDIsTotal = (data) => {
    try {
        let totalSum = 0, IMSSSum = 0, ISRSum = 0;

        data.map((row) => {
            if (row.Total) totalSum += parseFloat(row.Total);

            if (row.IMSS) {
                IMSSSum += parseFloat(row.IMSS);
            }
            if (row.ISR) {
                ISRSum += parseFloat(row.ISR);
            }
        });

        const result = { NumeroDeCFDIS: data.length, totalSum: totalSum.toFixed(2), IMSSSum: IMSSSum.toFixed(2), ISRSum: ISRSum.toFixed(2) };
        return result;
    } catch (e) {
        console.log(e);
    }
}

const extractAllPDFsPagesText = async (documentObjectArray) => {
    const promises = documentObjectArray.map(async (data) => {
        var mergedData = mergeValues(data);
        var key = mergedData["Delegación IMSS"] + ' / ' + mergedData["SubDelegación IMSS"];
        var combinedText = await combinePdfAllPagesText(mergedData, data.length);
        return { name: key, combinedText: combinedText }
    });

    return Promise.all(promises).then((data) => Promise.resolve(data));
}

const combinePdfAllPagesText = (data, length) => {
    return new Promise((resolve, reject) => {
        let text = '';
        try {
            for (let i = 1; i <= length; i++) {
                text += data['page' + i];
            }
            resolve(text);
        } catch (e) {
            reject(false);
        }
    });

}

const uploadFiles = (files, year, month, isRecurring, directoryId, documentType, subDocumentType) => {
    return new Promise(async (resolve, reject) => {
        try {
            let documentIds = [];
            let directoryDocumentIds = {};
            const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
            const provider = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();

            uploadFileOneByOne = files.map(async (file, index) => {
                try {
                    const documentUniqueId = uuidv4();
                    let fileName = "";
                    if (isRecurring.toLowerCase() == "true") {
                        fileName = `${directoryId}/Documentación Mensual/${year}/${month}/${documentType.toLowerCase()}/${documentUniqueId}_${file.name.replace(/\.[^/.]+$/, "")}`;
                    } else {
                        fileName = `${directoryId}/${documentType.toLowerCase()}/${documentUniqueId}_${file.name.replace(/\.[^/.]+$/, "")}`;
                    }

                    const fileInformation = {
                        path: file.path,
                        name: fileName,
                        contentType: file.documentType
                    };

                    const documentData = await extractInformationFromDocument(documentType, file);

                    let docName = file.name.replace(/\.[^/.]+$/, "");
                    // validate the document
                    const validate = await ValidateDocumentData(year, month, provider, directoryId, documentType, isRecurring, subDocumentType, documentData, docName);

                    if (validate) {
                        const responseFromS3 = await uploadFilesTos3(fileInformation);

                        //Document Logic, Upload Document
                        let type = (subDocumentType) ? subDocumentType : documentType;
                        let documentResponse = await DocumentCreation(type, responseFromS3, file, documentUniqueId, documentData);
                        if (documentResponse) documentIds.push(objectId(documentResponse));
                        if (documentResponse) {
                            if (documentType == 'CFDIs') {
                                let dataObject = JSON.parse(documentData);

                                if (dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"]) {
                                    let nomina12 = dataObject["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"];
                                    let FechaPagoDate = new Date(nomina12[0].$.FechaPago);

                                    year = FechaPagoDate.getFullYear();
                                    month = (FechaPagoDate.getMonth() + 1);
                                }
                            }

                            let key = year + '_' + month;

                            if (key in directoryDocumentIds) {
                                directoryDocumentIds[key].documentIds.push(documentResponse);
                            } else {
                                directoryDocumentIds[key] = { year: year, month: month, documentIds: [documentResponse] };
                            }
                        }

                        // Cleanup Locally
                        cleanUpUpload(file.path);
                    }
                } catch (e) {
                    console.log(e);
                    reject(e);
                }
            });

            // wait for upload all files
            const execUploadFileOneByOnePromise = await Promise.all(uploadFileOneByOne);
            // Update Directory by adding new document IDs
            const directoryDocIdsList = Object.keys(directoryDocumentIds).map(async (key) => {
                let { year, month, documentIds } = directoryDocumentIds[key];
                // Update Directory by adding new document IDs
                const directoryUpdation = await DirectoryUpdation(year, month, directoryId, documentType, isRecurring, documentIds, subDocumentType);
            });

            await Promise.all(directoryDocIdsList);
            resolve(true);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const ValidateDocumentData = (year, month, provider, directoryId, documentType, isRecurring, subDocumentType, documentData, docName) => {
    // console.log(documentData);
    return new Promise(async (resolve, reject) => {
        try {
            if (["Autorización STPS", "Contrato", "Documentación Corporativa"].includes(documentType)) {
                resolve(true);
            } else {
                let data;

                if (documentType == "CFDIs") data = JSON.parse(documentData);
                else data = mergeValues(JSON.parse(documentData));
                // console.log(data);
                if (!Object.keys(data).length) {
                    reject("No pudimos leer el PDF por favor adjunte su descarga original del SAT");
                } else {
                    let monthName = meses.find(_mes => _mes.id == month).nombre;
                    let monthNameLower = monthName.toLowerCase();
                    if (documentType == "CFDIs") {

                        if (data["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"]) {
                            let nomina12 = data["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"];
                            let FechaPagoDate = new Date(nomina12[0].$.FechaPago);

                            year = FechaPagoDate.getFullYear();
                            month = (FechaPagoDate.getMonth() + 1);
                        }

                        if (data["cfdi:Comprobante"].$.TipoDeComprobante == "N") {
                            // Check Same UUID file uploaded or not
                            let UUID = data["cfdi:Comprobante"]["cfdi:Complemento"][0]["tfd:TimbreFiscalDigital"][0].$.UUID;
                            let cfdiDocumentCheck = await Document.find({ isValid: true, DocumentType: documentType, 'DocumentData': new RegExp('"UUID":"' + UUID + '"', 'i') });

                            if (cfdiDocumentCheck.length > 0) {
                                let storeFileStatus = await saveZipExtractedDocumentDetail(year, month, directoryId, null, documentType, isRecurring, '', docName, true, true, 'UUID document already exists');
                                resolve(false);
                            } else {
                                let storeFileStatus = await saveZipExtractedDocumentDetail(year, month, directoryId, null, documentType, isRecurring, '', docName, true, true, '');
                            } resolve(true);
                        } else {
                            let storeFileStatus = await saveZipExtractedDocumentDetail(year, month, directoryId, null, documentType, isRecurring, '', docName, true, true, 'File doesn\'t have TipoDeComprobante="N"');
                            resolve(false);
                        }

                    } else if (documentType == 'Declaración Retención ISR') {
                        if (data.RFC == provider.rfc && data.name == "Declaración Provisional o Definitiva de Impuestos Federales" && (data['Período de la declaración'] == monthName || data['Período de la declaración'] == monthNameLower) && data.Ejercicio == year) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Pago Retención ISR") {
                        if (data.RFC == provider.rfc && data.name == "DECLARACIÓN PROVISIONAL O DEFINITIVA DE IMPUESTOS FEDERALES" && (data["Período de la declaración"] == monthName || data["Período de la declaración"] == monthNameLower) && data.Ejercicio == year) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Retención IVA") {
                        if (data['Período de la declaración']) data.PERIODO = data['Período de la declaración'];
                        if (data.Ejercicio) data.EJERCICIO = data.Ejercicio;

                        if (data.RFC == provider.rfc && (data.name).toLowerCase() == ("DECLARACIÓN PROVISIONAL O DEFINITIVA DE IMPUESTOS FEDERALES").toLowerCase() && (data.PERIODO == monthName || data.PERIODO == monthNameLower) && data.EJERCICIO == year) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Pago IVA") {
                        if (data.RFC == provider.rfc && (data["Período de la declaración"] == monthName || data["Período de la declaración"] == monthNameLower) && data.Ejercicio == year) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "SUA IMSS") {
                        if (data["Período de Proceso"]) {
                            let dateArr = data["Período de Proceso"].split('-');
                            data.Periodo = dateArr[0];
                            data.Ejercicio = dateArr[1];
                        } else if (data["Bimestre de Proceso"]) {
                            let dateArr = data["Bimestre de Proceso"].split('-');
                            data.Periodo = dateArr[0];
                            data.Ejercicio = dateArr[1];
                        } else {
                            data.Periodo = " ";
                            data.Ejercicio = " ";
                        }

                        if (data.RFC == provider.rfc && (data.Periodo == monthName || data.Periodo == monthNameLower) && data.Ejercicio == year) {
                            resolve(true);
                        } else reject(false);

                    } else if (documentType == "PAGO IMSS") {
                        if (data['Fecha de Pago']) {
                            let date = (data['Fecha de Pago'].includes('/')) ? data['Fecha de Pago'].split('/') : data['Fecha de Pago'].split('-');
                            if (date.length == 3 && date[2] >= year) resolve(true);
                            else reject(false);
                        } else if (data['Periodo de Pago']) {
                            let date = data['Periodo de Pago'].split(' ');
                            if (date.length == 4 && date[3] >= year) resolve(true);
                            else reject(false);
                        } else if (data['ano']) {
                            if (data['ano'] >= year) resolve(true);
                            else reject(false);
                        } else if (data['Periodo de pago']) {
                            if (data['Periodo de pago'].length == 6 && parseInt(data['Periodo de pago'].substring(0, 4)) >= year) resolve(true);
                            else reject(false);
                        } else reject(false);
                    } else if (documentType == "Opinión SAT") {
                        if (data["Clave de R.F.C."] == provider.rfc && data.name == "Opinión del cumplimiento de obligaciones fiscales" && data.dateText.includes(year) && (data.dateText.includes(monthName) || data.dateText.includes(monthNameLower))) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Opinión IMSS") {
                        if (data["Clave de R.F.C."] == provider.rfc && (data.name == "Opinión del Cumplimiento de Obligaciones en materia de Seguridad Social" || data.name == "Opinión del Cumplimiento de Obligaciones Fiscales en materia de Seguridad Social") && data["dateText"].includes(year) && (data["dateText"].includes(monthName) || data.dateText.includes(monthNameLower))) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Opinión INFONAVIT") {
                        if (data.name.includes("Asunto Constancia de Situación Fiscal") && data.issuedFrom.includes("Instituto del Fondo Nacional de la Vivienda para los Trabajadores") && data["dateText"].includes(year) && (data["dateText"].includes(monthName) || data["dateText"].includes(monthNameLower))) {
                            resolve(true);
                        } else reject(false);
                    } else if (documentType == "Resumen de Liquidación SUA") {
                        if (subDocumentType == "Resumen de Liquidación SUA 1st vers") {
                            if (data["dateText"].includes(year) && (data["dateText"].includes(monthName) || data["dateText"].includes(monthNameLower))) {
                                resolve(true);
                            } else reject(false);
                        } else if (subDocumentType == "Resumen de Liquidación SUA 2nd vers") {
                            if (data["dateText"].includes(year) && data["dateText"].includes(month)) {
                                resolve(true);
                            } else reject(false);
                        } else reject(false);
                    } else {
                        reject(false);
                    }
                }
            }
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const getClientName = async (id) => {
    const DirectoryAccountMap = await model('accountsRelations').findOne({ providerId: objectId(id) }).lean();

    if (!DirectoryAccountMap) return '';

    const client = await User.findOne({ _id: objectId(DirectoryAccountMap.clientId) }).lean();

    if (client) return client.name;
    else return '';
}

const saveZipDocumentDetail = (year, month, directory_id, documentType, isRecurring, fileInformation) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Create a New Document
            let doc = new ZipDocument();
            doc.directory = directory_id;
            doc.documentType = documentType;
            doc.year = year;
            doc.month = month;
            doc.isRecurring = isRecurring;
            doc.path = fileInformation.path;

            doc.save().then((documentDetails) => {
                resolve(documentDetails);
            }).catch((e) => {
                reject(false);
            })
        } catch (e) {
            reject(false);
        }
    });
}

const updateZipDocumentStatus = (_id) => {
    return new Promise((resolve, reject) => {
        try {
            ZipDocument.findById(objectId(_id)).then((detail) => {
                detail.isExtracted = true;
                detail.save().then((docDetail) => {
                    resolve(docDetail);
                }).catch((e) => {
                    reject(false);
                });
            }).catch((e) => {
                reject(false);
            });
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const updateZipExtractedDocumentStatus = (_id, error = '') => {
    return new Promise((resolve, reject) => {
        try {
            ZipExtractedDocument.findById(objectId(_id)).then((detail) => {
                detail.isDataExtracted = true;
                detail.error = error;
                detail.save().then((docDetail) => {
                    resolve(docDetail);
                }).catch((e) => {
                    reject(false);
                });
            }).catch((e) => {
                reject(false);
            });
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const saveZipExtractedDocumentDetail = (year, month, directory_id, documentZip_id, documentType, isRecurring, filePath, fileName, isDataExtracted = false, isSelected = false, error = '') => {
    return new Promise(async (resolve, reject) => {
        try {
            // Create a New Document
            let doc = new ZipExtractedDocument();
            doc.directory = directory_id;
            doc.zipDocument = documentZip_id;
            doc.documentType = documentType;
            doc.year = year;
            doc.month = month;
            doc.isRecurring = isRecurring;
            doc.name = fileName.replace(/\.[^/.]+$/, "");
            doc.path = filePath;
            doc.contentType = "text/xml";
            doc.error = error;
            doc.isDataExtracted = isDataExtracted;
            doc.isSelected = isSelected;

            doc.save().then((documentDetails) => {
                resolve(documentDetails);
            }).catch((e) => {
                console.log(e);
                reject(false);
            })
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const checkAutorizacionSTPSDocStatus = (documents, directoryId) => {
    return new Promise(async (resolve, reject) => {
        try {
            let AutorizacionSTPS = JSON.parse(documents[0]["DocumentData"])[0];
            const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
            const Provider = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();
            let isValid = false;
            if (AutorizacionSTPS.STPS) {
                const dateTextArr = AutorizacionSTPS.dateText.split(' ');
                if (dateTextArr.length == 10) {
                    let day = dateTextArr[5];
                    let month = meses.find(_mes => _mes.nombre == dateTextArr[7]).id;
                    let year = dateTextArr[9];
                    AutorizacionSTPS.date = `${month}/${day}/${year}`;

                    // Checking for Autorización STPS PDF validity
                    const currentDate = new Date();
                    const AutorizacionSTPSDate = new Date(`${month}/${day}/${parseInt(year) + 3}`);

                    if (AutorizacionSTPSDate.getTime() > currentDate.getTime() && AutorizacionSTPS.RFC == Provider.rfc) {
                        isValid = true;
                    }
                }
            }
            resolve(isValid);
        } catch (e) {
            reject(false);
        }
    });
}

const checkDirectoryDataStatus = (year, month, directoryData) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Year Index
            const yearIndex = findIndexOnObject(year, directoryData.toJSON().documents["reccuring"], "year");
            // This is the Month Index
            const monthIndex = findIndexOnObject(month, directoryData.toJSON().documents["reccuring"][yearIndex].months, "month");
            // This depends on document Type
            const monthDocuments = directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents;
            let response = {};
            const monthDoc = monthDocuments.map(async (documents) => {

                const folderName = documents.name;
                const documentTypeIndex = findIndexOnObject(folderName, directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents, "name");

                if (folderName == "Resumen de Liquidación SUA") {
                    const subDocs = directoryData.toJSON().documents.reccuring[yearIndex].months[monthIndex].documents[documentTypeIndex].documents;
                    const subDocuments = subDocs.map(async (subDocument) => {
                        const subDocumentTypeIndex = findIndexOnObject(subDocument.name, directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents[documentTypeIndex].documents, "name");

                        const options = {
                            '_id': { $in: directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents[documentTypeIndex].documents[subDocumentTypeIndex].value },
                            isValid: true
                        }

                        const docs = await Document.find(options);
                        response[subDocument.name] = docs;
                    });

                    await Promise.all(subDocuments);
                } else {
                    const options = {
                        '_id': { $in: directoryData.toJSON().documents.reccuring[yearIndex].months[monthIndex].documents[documentTypeIndex].value },
                        isValid: true
                    }

                    const docs = await Document.find(options);
                    response[folderName] = docs;
                }
            });

            await Promise.all(monthDoc);
            const status = await checkByComparingDirectoryData(response);

            resolve(status);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const checkByComparingDirectoryData = (data) => {
    return new Promise(async (resolve, reject) => {
        let result = { CFDIs: { isValid: false }, DeclaracionRetencionISR: { isValid: false }, PagoRetencionISR: { isValid: false }, RetencionIVA: { isValid: false }, PagoIVA: { isValid: false }, SUA_IMSS: { isValid: false }, PagoIMSS: { isValid: false }, OpinionSAT: { isValid: false }, OpinionIMSS: { isValid: false }, OpinionINFONAVIT: { isValid: false }, resumendeLiquidacionSUA: { isValid: false } };
        try {
            const formattedData = data.CFDIs.map((row) => CFDIJSONFormatHelper(row.DocumentData));

            await Promise.all(formattedData);
            const CFDIsTotal = calculateCFDIsTotal(formattedData);
            result.CFDIs.isValid = (data.CFDIs.length) ? true : false;

            if (data['Declaración Retención ISR'].length) {
                let DeclaracionRetencionISR = JSON.parse(data['Declaración Retención ISR'][0].DocumentData);
                result.DeclaracionRetencionISR.isValid = true;
            }

            if (data['Pago Retención ISR'].length) {
                result.PagoRetencionISR.isValid = true;
            }

            if (data['Retención IVA'].length) {
                result.RetencionIVA.isValid = true;
            }

            if (data['Pago IVA'].length) {
                result.PagoIVA.isValid = true;
            }

            if (data['SUA IMSS'].length > 0) {
                const suaImssFormattedData = data['SUA IMSS'].map((row) => JSON.parse(row.DocumentData));
                await Promise.all(suaImssFormattedData);
                // Combine all PDF pages data in single text to check CURP
                let PDFsCombinedTextArray = await extractAllPDFsPagesText(suaImssFormattedData);

                // Checking all CURPS include in PDF
                let numOfCurpsFound = 0;
                let checkForCurps = formattedData.map((row) => {
                    let check = PDFsCombinedTextArray.some((TextObj) => {
                        return TextObj.combinedText.includes(row.CURP);
                    });

                    if (check) numOfCurpsFound++;
                });
                // Wait till loop complete
                const resolvePromise = await Promise.all(checkForCurps);
                result.SUA_IMSS.isValid = (numOfCurpsFound == data.CFDIs.length);
            }

            if (data['PAGO IMSS'].length) {
                result.PagoIMSS.isValid = true;
            }

            if (data['Opinión SAT'].length) {
                let OpinionSAT = mergeValues(JSON.parse(data['Opinión SAT'][0].DocumentData));
                result.OpinionSAT.isValid = OpinionSAT.POSITIVO;
            }

            if (data['Opinión IMSS'].length) {
                let OpinionIMSS = mergeValues(JSON.parse(data['Opinión IMSS'][0].DocumentData));
                result.OpinionIMSS.isValid = OpinionIMSS.opinionIsPositive;
            }

            if (data['Opinión INFONAVIT'].length) {
                let OpinionINFONAVIT = mergeValues(JSON.parse(data['Opinión INFONAVIT'][0].DocumentData));
                result.OpinionINFONAVIT.isValid = OpinionINFONAVIT.opinionIsPositive;
            }

            if (data['Resumen de Liquidación SUA 1st vers'].length && data['Resumen de Liquidación SUA 2nd vers'].length) {
                let firstTotal = 0, secondTotal = 0;

                if (data['Resumen de Liquidación SUA 1st vers']) {
                    let firstData = mergeValues(JSON.parse(data['Resumen de Liquidación SUA 1st vers'][0].DocumentData));
                    firstTotal = parseFloat(firstData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (data['Resumen de Liquidación SUA 2nd vers']) {
                    let secondData = mergeValues(JSON.parse(data['Resumen de Liquidación SUA 2nd vers'][0].DocumentData));
                    secondTotal = parseFloat(secondData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (firstTotal && secondTotal && (firstTotal == secondTotal)) result.resumendeLiquidacionSUA.isValid = true;
            }

            resolve(result);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}

const chunkArray = (array, size) => {
    return new Promise((resolve, reject) => {
        try {
            const chunked_arr = [];
            let index = 0;
            while (index < array.length) {
                chunked_arr.push(array.slice(index, size + index));
                index += size;
            }
            resolve(chunked_arr);
        } catch (e) {
            reject([]);
        }
    });
}

const processDocumentsArray = (documentDetails) => {
    return new Promise(async (resolve, reject) => {
        try {
            let directoryDocumentIds = {};

            const documentList = documentDetails.map(async (document) => {
                let { _id, directory, year, month, documentType, isRecurring, name, path, contentType } = document;
                try {
                    // const documentUniqueId = uuidv4();
                    const documentUniqueId = (path.split("/").pop()).split('_')[0];

                    let fileName = `${directory}/Documentacion_Mensual/${year}/${month}/${documentType.toLowerCase()}/${documentUniqueId}_${name.replace(/\.[^/.]+$/, "")}`;

                    let fileInformation = {
                        path: path,
                        name: fileName,
                        contentType: contentType,
                        type: contentType
                    };

                    // This is Documentation Parsing, To extract data from the documents XMLs, PDFs etc and then put it in a string for later distribution.
                    const documentData = await extractInformationFromDocument(documentType, fileInformation);

                    const parseData = JSON.parse(documentData);
                    if (documentType == "CFDIs" && parseData["cfdi:Comprobante"].$.TipoDeComprobante == "N") {
                        // Check Same UUID file uploaded or not
                        let UUID = parseData["cfdi:Comprobante"]["cfdi:Complemento"][0]["tfd:TimbreFiscalDigital"][0].$.UUID;
                        let cfdiDocumentCheck = await Document.find({ isValid: true, DocumentType: documentType, 'DocumentData': new RegExp('"UUID":"' + UUID + '"', 'i') });

                        if (cfdiDocumentCheck.length > 0) {
                            // Cleanup Locally
                            cleanUpUpload(path);

                            const updateStatus = await updateZipExtractedDocumentStatus(_id, 'UUID document already exists');
                        } else {
                            // const responseFromS3 = await uploadFilesTos3(fileInformation);
                            const responseFromS3 = { key: fileName, Key: fileName };

                            if (responseFromS3) {
                                fileInformation.name = name;
                                //Document Logic, Upload Document, Update Directory
                                const documentResponse = await DocumentCreation(documentType, responseFromS3, fileInformation, documentUniqueId, documentData);

                                // Cleanup Locally
                                cleanUpUpload(path);

                                // Upload file in specific folder by document payment date
                                if (parseData["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"]) {
                                    let nomina12 = parseData["cfdi:Comprobante"]["cfdi:Complemento"][0]["nomina12:Nomina"];
                                    let FechaPagoDate = new Date(nomina12[0].$.FechaPago);

                                    year = FechaPagoDate.getFullYear();
                                    month = (FechaPagoDate.getMonth() + 1);
                                }

                                if (documentResponse) {
                                    let key = directory + '_' + year + '_' + month;
                                    if (key in directoryDocumentIds) {
                                        directoryDocumentIds[key].documentIds.push(documentResponse);
                                    } else {
                                        directoryDocumentIds[key] = { directory: directory, year: year, month: month, documentType: documentType, isRecurring: (isRecurring) ? "true" : "false", documentIds: [documentResponse] };
                                    }
                                    const updateStatus = await updateZipExtractedDocumentStatus(_id);
                                }
                            }
                        }
                    } else {
                        // Cleanup Locally
                        cleanUpUpload(path);

                        const updateStatus = await updateZipExtractedDocumentStatus(_id, 'File doesn\'t have TipoDeComprobante="N"');
                    }
                } catch (e) {
                    // Cleanup Locally
                    cleanUpUpload(path);

                    console.log(e);
                    const updateStatus = await updateZipExtractedDocumentStatus(_id, 'Data not extracted');
                }
            });

            await Promise.all(documentList);
            resolve(directoryDocumentIds);
        } catch (e) {
            console.log(e);
            reject([]);
        }
    })
}

const insertDocumentIdsToDirectory = (directoryDocumentIds) => {
    return new Promise(async (resolve, reject) => {
        try {
            const pdffileArray = Object.keys(directoryDocumentIds);
            for (let i = 0; i < pdffileArray.length; i++) {
                let { directory, year, month, documentType, isRecurring, documentIds } = directoryDocumentIds[pdffileArray[i]];
                await DirectoryUpdation(year, month, directory, documentType, isRecurring, documentIds);
            }
            resolve(true);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const MonthFolderLockstatus = (directoryId, year, month, islockedstatus, userId, notification_type) => {
    return new Promise(async (resolve, reject) => {
        try {
            Directory.findById(objectId(directoryId)).then((directoryData) => {
                // If no response for the directory Id, Return False promise
                if (!directoryData) {
                    reject(false)
                }
                const yearIndex = findIndexOnObject(year, directoryData.documents["reccuring"], "year");
                const monthIndex = findIndexOnObject(month, directoryData.documents["reccuring"][yearIndex].months, "month");
                directoryData.documents.reccuring[yearIndex].months[monthIndex].islocked = islockedstatus;

                directoryData.save().then((directory) => {
                    if (!directory) {
                        reject(false);
                    }
                    resolve(directoryData);
                    AddNotification(directoryId, year, month, userId, notification_type);
                }).catch((err) => {
                    // Console the Error
                    console.log(err);
                    reject(false)
                });
            }).catch((err) => {
                // Console the Error
                console.log(err);
                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}

const uploadClientFiles = (file, id) => {
    return new Promise(async (resolve, reject) => {
        let responseFromS3 = "";
        try {
            const documentUniqueId = uuidv4();
            let fileName = "";
            fileName = `faturas/${id}/${documentUniqueId}_${file.name}`;
            const fileInformation = {
                path: file.path,
                name: fileName,
                contentType: file.documentType
            };
            responseFromS3 = await uploadFilesTos3(fileInformation);
            if (responseFromS3) {
                const updateStatus = await updateFilePathInFacturas(id, responseFromS3, fileInformation);
                const filepathnew = await getFilesFromS3(updateStatus.clientfilepath);
                let sendfilemail = await sendfacturasfileuploademail(updateStatus, filepathnew);
            }
            cleanUpUpload(file.path);
            resolve(responseFromS3);

        } catch (e) {
            console.log(e);
            reject(false);
        }
        resolve(responseFromS3);

    });
}

const updateFilePathInFacturas = (id, responseFromS3, fileInformation) => {
    return new Promise(async (resolve, reject) => {
        try {
            Facturas.findById(objectId(id)).then((facturasData) => {
                if (!facturasData) {
                    reject(false)
                }
                facturasData.clientfilepath = responseFromS3.key;
                facturasData.save().then((faturas) => {
                    if (!faturas) {
                        reject(false);
                    }
                    resolve(faturas);
                }).catch((err) => {
                    console.log(err);
                    reject(false)
                });
            }).catch((err) => {
                console.log(err);
                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}


const deleteClientFacturasFiles = (id) => {
    return new Promise(async (resolve, reject) => {
        try {
            Facturas.findById(objectId(id)).then((facturasData) => {
                if (!facturasData) {
                    reject(false)
                }

                facturasData.clientfilepath = "";

                facturasData.save().then((faturas) => {
                    if (!faturas) {
                        reject(false);
                    }
                    resolve(faturas);
                }).catch((err) => {
                    // Console the Error
                    console.log(err);
                    reject(false)
                });
            }).catch((err) => {
                // Console the Error
                console.log(err);
                reject(false)
            });
        } catch (e) {
            reject(false);
        }
    });
}

const sendfacturasfileuploademail = function (facturasData, filepath) {
    const Montharray = ["Enero", "Febrero", "Marzo", "Abril", "Mayo", "Junio", "Julio", "Agosto", "Septiembre", "Octubre", "Noviembre", "Diciembre"];
    const mailgun = new Mailgun({ apiKey: process.env.MAIL_API, domain: process.env.MAIL_DOMAIN });
    const fromEmail = 'no-reply@kentaurus.com.mx';

    let monthno = parseInt(facturasData.month) + 1;
    facturasData.month = Montharray[monthno];
    facturasData.amount = formatter.format(facturasData.amount);
    User.findById(objectId(facturasData.clientId)).then((userdata) => {
        Twig.renderFile('./templates/emails/facturasfileupload.html.twig', { month: facturasData.month, year: facturasData.year, client: userdata.name, amount: facturasData.amount, filepath: filepath }, (err, html) => {

            const data = {
                from: fromEmail,
                to: 'testinvoiceadmin@dispostable.com',
                // to: 'contacto@ebitsa.com',
                subject: 'Payment File Uploaded by Client, EBITSA',
                html: html
            }

            // MailGun Fire Function

            mailgun.messages().send(data, function (err, body) {
                // If there is an error, render the error page
                if (err) {
                    console.log("got an error on mailgun: ", err);
                    return false;
                }
                // Else we can greet and leave
                else {
                    console.log(`Mail Send, To the user `);
                    return true;
                }
            });
        });
    });




}
const formatter = new Intl.NumberFormat('en-US', {
    style: 'currency',
    currency: 'USD',
});

const getNotificationTime = function (targetTimestamp) {
    let currentDate = new Date();
    let currentTimeInms = currentDate.getTime();
    let targetDate = new Date(targetTimestamp);
    let targetTimeInms = targetDate.getTime();
    let elapsed = Math.floor((currentTimeInms - targetTimeInms) / 1000);
    if (elapsed < 1) {
        return '0s';
    }
    if (elapsed < 60) { //< 60 sec
        return `Hace ${elapsed} segundos`;
    }
    if (elapsed < 3600) { //< 60 minutes
        return `Hace ${Math.floor(elapsed / (60))} minutos`;
    }
    if (elapsed < 86400) { //< 24 hours
        return `Hace ${Math.floor(elapsed / (3600))} horas`;
    }
    if (elapsed < 604800) { //<7 days
        return `Hace ${Math.floor(elapsed / (86400))} días`;
    }
    if (elapsed < 2628000) { //<1 month
        return `${targetDate.getDate()} ${MonthNames[targetDate.getMonth()]}`;
    }
    return `${targetDate.getDate()} ${MonthNames[targetDate.getMonth()]} ${targetDate.getFullYear()}`; //more than a monh

}

const getdirectoryDetailsbyProvider = (provider, userId) => {
    return new Promise(async (resolve, reject) => {
        try {
            const DirectoryAccountMap = await model('accountsRelations').findOne({ providerId: objectId(provider), clientId: objectId(userId) }).lean();
            if (DirectoryAccountMap) {
                const providername = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();
                const data = {
                    "directoryId": DirectoryAccountMap.directory,
                    "providername": providername.name
                }
                resolve(data);

            } else {
                reject(false);
            }
        } catch (err) {
            reject(false);
        }
    });
}



const ReportDirectoryStatus = (year, month, directoryData, folders) => {
    return new Promise(async (resolve, reject) => {
        try {
            // Year Index
            const yearIndex = findIndexOnObject(year, directoryData.toJSON().documents["reccuring"], "year");
            // This is the Month Index
            const monthIndex = findIndexOnObject(month, directoryData.toJSON().documents["reccuring"][yearIndex].months, "month");
            console.log("month", month);
            // This depends on document Type
            // const monthDocuments = directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex];
            let response = {};
            if (monthIndex) {
                const monthDocuments = directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents;
                const monthDoc = monthDocuments.map(async (documents) => {
                    const folderName = documents.name;
                    const checkfolder = folders.includes(folderName);
                    const documentTypeIndex = findIndexOnObject(folderName, directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents, "name");
                    if (checkfolder) {
                        if (folderName == "Resumen de Liquidación SUA") {
                            const subDocs = directoryData.toJSON().documents.reccuring[yearIndex].months[monthIndex].documents[documentTypeIndex].documents;
                            const subDocuments = subDocs.map(async (subDocument) => {
                                const subDocumentTypeIndex = findIndexOnObject(subDocument.name, directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents[documentTypeIndex].documents, "name");

                                const options = {
                                    '_id': { $in: directoryData.toJSON().documents["reccuring"][yearIndex].months[monthIndex].documents[documentTypeIndex].documents[subDocumentTypeIndex].value },
                                    isValid: true
                                }

                                const docs = await Document.find(options);
                                response[subDocument.name] = docs;
                            });

                            await Promise.all(subDocuments);
                        } else {
                            const options = {
                                '_id': { $in: directoryData.toJSON().documents.reccuring[yearIndex].months[monthIndex].documents[documentTypeIndex].value },
                                isValid: true
                            }

                            const docs = await Document.find(options);
                            response[folderName] = docs;
                        }
                    }
                });
                await Promise.all(monthDoc);
                const status = await ReportFolderStatusByComparingDirectoryData(response);
                resolve(status);
            } else {
                folders.forEach(folders => {
                    response[folders] = "";
                });
                const status = await ReportFolderStatusByComparingDirectoryData(response);
                resolve(status);
            }
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const ReportFolderStatusByComparingDirectoryData = (data) => {
    console.log("data", data);
    return new Promise(async (resolve, reject) => {
        let result = {};

        try {
            if (data.hasOwnProperty("CFDIs") && data['CFDIs'].length) {
                const formattedData = data.CFDIs.map((row) => CFDIJSONFormatHelper(row.DocumentData));

                await Promise.all(formattedData);
                const CFDIsTotal = calculateCFDIsTotal(formattedData);
                result.CFDIs = (data.CFDIs.length) ? 'complete' : 'no_document';
            } else {
                if (data.hasOwnProperty("CFDIs")) {
                    result.CFDIs = 'no_document';
                }
            }

            if (data.hasOwnProperty("Declaración Retención ISR") && data['Declaración Retención ISR'].length) {
                let PagoRetenciónISR = JSON.parse(data['Declaración Retención ISR'][0].DocumentData);
                result.DeclaraciónRetenciónISR = 'complete';
            } else {
                if (data.hasOwnProperty("Declaración Retención ISR")) {
                    result.DeclaraciónRetenciónISR = 'no_document';
                }
                // delete result.DeclaraciónRetenciónISR;
            }

            if (data.hasOwnProperty("Pago Retención ISR") && data['Pago Retención ISR'].length) {
                result.PagoRetenciónISR = 'complete';
            } else {
                if (data.hasOwnProperty("Pago Retención ISR")) {
                    result.PagoRetenciónISR = 'no_document';
                }
                // delete result.PagoRetenciónISR;
            }

            if (data.hasOwnProperty("Retención IVA") && data['Retención IVA'].length) {
                result.RetenciónIVA = 'complete';
            } else {
                if (data.hasOwnProperty("Retención IVA")) {
                    result.RetenciónIVA = 'no_document';
                }
                // delete result.RetenciónIVA;
            }

            if (data.hasOwnProperty("Pago IVA") && data['Pago IVA'].length) {
                result.PagoIVA = 'complete';
            } else {
                if (data.hasOwnProperty("Pago IVA")) {
                    result.PagoIVA = 'no_document';
                }
                // delete result.PagoIVA;
            }

            if (data.hasOwnProperty("SUA IMSS") && data['SUA IMSS'].length > 0) {
                const suaImssFormattedData = data['SUA IMSS'].map((row) => JSON.parse(row.DocumentData));
                await Promise.all(suaImssFormattedData);
                // Combine all PDF pages data in single text to check CURP
                let PDFsCombinedTextArray = await extractAllPDFsPagesText(suaImssFormattedData);

                // Checking all CURPS include in PDF
                let numOfCurpsFound = 0;
                let checkForCurps = formattedData.map((row) => {
                    let check = PDFsCombinedTextArray.some((TextObj) => {
                        return TextObj.combinedText.includes(row.CURP);
                    });

                    if (check) numOfCurpsFound++;
                });
                // Wait till loop complete
                const resolvePromise = await Promise.all(checkForCurps);
                result.SUAIMSS = (numOfCurpsFound == data.CFDIs.length) ? 'complete' : 'no_document';
                // result.SUAIMSS='complete';
            } else {
                if (data.hasOwnProperty("SUA IMSS")) {
                    result.SUAIMSS = 'no_document';
                }
                // delete result.SUAIMSS;
            }

            if (data.hasOwnProperty("PAGO IMSS") && data['PAGO IMSS'].length) {
                result.PAGOIMSS = 'complete';
            } else {
                if (data.hasOwnProperty("PAGO IMSS")) {
                    result.PAGOIMSS = 'no_document';
                }
                // delete result.PAGOIMSS;
            }

            if (data.hasOwnProperty("Opinión SAT") && data['Opinión SAT'].length) {
                let OpinionSAT = mergeValues(JSON.parse(data['Opinión SAT'][0].DocumentData));
                result.OPINIÓNSAT = OpinionSAT.POSITIVO ? 'complete' : 'no_document';
            } else {
                if (data.hasOwnProperty("Opinión SAT")) {
                    result.OPINIÓNSAT = 'no_document';
                }
                // delete result.OPINIÓNSAT;
            }

            if (data.hasOwnProperty("Opinión IMSS") && data['Opinión IMSS'].length) {
                let OpinionIMSS = mergeValues(JSON.parse(data['Opinión IMSS'][0].DocumentData));
                result.OPINIÓNIMSS = OpinionIMSS.opinionIsPositive ? 'complete' : 'no_document';
            } else {
                if (data.hasOwnProperty("Opinión IMSS")) {
                    result.OPINIÓNIMSS = 'no_document';
                }
                // delete result.OPINIÓNIMSS;
            }

            if (data.hasOwnProperty("Opinión INFONAVIT") && data['Opinión INFONAVIT'].length) {
                let OpinionINFONAVIT = mergeValues(JSON.parse(data['Opinión INFONAVIT'][0].DocumentData));
                result.OpiniónINFONAVIT = OpinionINFONAVIT.opinionIsPositive ? 'complete' : 'no_document';
            } else {
                if (data.hasOwnProperty("Opinión INFONAVIT")) {
                    result.OpiniónINFONAVIT = 'no_document';
                }
                // delete result.OpiniónINFONAVIT;
            }

            if (data.hasOwnProperty("Resumen de Liquidación SUA 1st vers") && data['Resumen de Liquidación SUA 1st vers'].length && data['Resumen de Liquidación SUA 2nd vers'].length) {
                let firstTotal = 0, secondTotal = 0;

                if (data.hasOwnProperty("Resumen de Liquidación SUA 1st vers") && data['Resumen de Liquidación SUA 1st vers']) {
                    let firstData = mergeValues(JSON.parse(data['Resumen de Liquidación SUA 1st vers'][0].DocumentData));
                    firstTotal = parseFloat(firstData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (data.hasOwnProperty("Resumen de Liquidación SUA 2nd vers") && data['Resumen de Liquidación SUA 2nd vers']) {
                    let secondData = mergeValues(JSON.parse(data['Resumen de Liquidación SUA 2nd vers'][0].DocumentData));
                    secondTotal = parseFloat(secondData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (firstTotal && secondTotal && (firstTotal == secondTotal)) {
                    result.ResumendeLiquidaciónSUA = 'complete';
                } else {
                    result.ResumendeLiquidaciónSUA = 'no_document';
                }
            } else {
                if (data.hasOwnProperty("Resumen de Liquidación SUA 1st vers") || data.hasOwnProperty("Resumen de Liquidación SUA 2nd vers")) {
                    result.ResumendeLiquidaciónSUA = 'no_document';
                }
                // delete result.ResumendeLiquidaciónSUA;
            }


            resolve(result);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


const checkreportFromDocument = (directoryId, year, month, documents, folders, provider_rfc) => {
    return new Promise((resolve, reject) => {
        try {
            let response = { "CFDIs": [], "SUA IMSS": [] };
            let tempArray = [];
            // Getting data from non recurring folder
            // console.log("folders",folders);
            documents.nonRecurring.forEach(async (documentData, index) => {
                const checkfolder = folders.includes(documentData.name);
                if (checkfolder) {
                    if (documentData.name === "Autorización STPS") {
                        nonRecurringPromises = documentData.value.map(async (documentId) => {
                            const documentObj = await Document.findOne({ _id: documentId }).lean();
                            if (documentObj && documentObj.isValid && documentObj["DocumentData"]) {
                                const data = JSON.parse(documentObj["DocumentData"]);
                                response[documentData.name] = data;
                            }
                        });
                    }

                    if (index == documents.nonRecurring.length - 1) {
                        const execNonRecurringPromise = await Promise.all(nonRecurringPromises);
                    }
                }

            });
            // Getting data from recurring folder
            documents.reccuring.forEach((documentData) => {
                if (documentData.year == year) {
                    documentData.months.map(async (monthsObject) => {
                        if (monthsObject.month == month) {
                            promises = monthsObject.documents.map(async (document, index) => {
                                const checkfolder = folders.includes(document.name);
                                if (checkfolder) {
                                    if (document.name == "Resumen de Liquidación SUA") {
                                        parseAllDoc = document.documents.map(async (subDocuments) => {
                                            parseSubDoc = subDocuments.value.map(async (documentId) => {
                                                const documentObj = await Document.findOne({ _id: documentId, isValid: true }).lean();
                                                if (documentObj) {
                                                    if (documentObj["DocumentData"]) {
                                                        const data = JSON.parse(documentObj["DocumentData"]);
                                                        response[subDocuments.name] = data;
                                                    }
                                                }
                                            });
                                            await Promise.all(parseSubDoc);
                                        });

                                        // Wait till all document data processed
                                        const execParseAllDoc = await Promise.all(parseAllDoc);
                                    } else {
                                        parseAllDoc = document.value.map(async (documentId) => {
                                            const documentObj = await Document.findOne({ _id: documentId, isValid: true }).lean();
                                            if (documentObj) {

                                                if (documentObj["DocumentType"] == "CFDIs") {
                                                    if (documentObj["DocumentData"]) {
                                                        const data = CFDIJSONFormatHelper(documentObj["DocumentData"]);
                                                        response[document.name].push(data);
                                                        // tempArray.push(data);
                                                        // const docname=document.name;
                                                        // response[docname]=tempArray;
                                                    }
                                                } else {
                                                    if (documentObj["DocumentData"]) {
                                                        const data = JSON.parse(documentObj["DocumentData"]);
                                                        if (documentObj["DocumentType"] == "SUA IMSS") response[document.name].push(data);
                                                        else response[document.name] = data;
                                                    }
                                                }
                                            }
                                        });
                                        // Wait till all document data processed
                                        const execParseAllDoc = await Promise.all(parseAllDoc);
                                    }
                                }
                            });

                            // Wait till all months' documents processed
                            Promise.all(promises).then(async () => {
                                const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
                                const Provider = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();
                                const Client = await User.findOne({ _id: objectId(DirectoryAccountMap.clientId) }).lean();
                                // console.log(response);
                                let result = await generateReportData(year, month, response, folders, Provider);
                                // console.log("res",result);
                                // console.log("checkobj",Object.keys(result).length);
                                // if(Object.keys(result).length ==0){
                                //     resolve();
                                // }

                                resolve(result);
                            });
                        }
                        // else{
                        //     const result = await generateReportData(year, month, response,folders);
                        //     resolve(result);
                        // }
                    });
                }
            });
            documents.reccuring.forEach((documentData) => {
                if (documentData.year == year) {
                    documentData.months.map(async (monthsObject) => {
                        if (monthsObject.month != month) {
                            const DirectoryAccountMap = await model('accountsRelations').findOne({ directory: objectId(directoryId) }).lean();
                            const Provider = await User.findOne({ _id: objectId(DirectoryAccountMap.providerId) }).lean();
                            const Client = await User.findOne({ _id: objectId(DirectoryAccountMap.clientId) }).lean();
                            // console.log(response);
                            let result = await generateReportData(year, month, response, folders, Provider);
                            resolve(result);

                        }
                    });

                }

            });


        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}
const generateReportData = (year, month, data, folders, provider) => {

    return new Promise(async (resolve, reject) => {
        let result = {};
        let monthName = meses.find(_mes => _mes.id == month).nombre;
        let monthNameLower = monthName.toLowerCase();
        try {
            if (data['CFDIs']) {
                const CFDIsTotal = calculateCFDIsTotal(data.CFDIs);
                result.CFDIs = CFDIsTotal;
                result.CFDIs.isValid = (data.CFDIs.length) ? true : false;
                result.CFDIs.isDocumentAvailable = (data.CFDIs.length) ? true : false;
            }

            if (data['Autorización STPS']) {
                let AutorizacionSTPS = data['Autorización STPS'][0];
                if (AutorizacionSTPS.STPS) {
                    const dateTextArr = AutorizacionSTPS.dateText.split(' ');
                    if (dateTextArr.length == 10) {
                        let day = dateTextArr[5];
                        let month = meses.find(_mes => _mes.nombre == dateTextArr[7]).id;
                        let year = dateTextArr[9];
                        AutorizacionSTPS.date = `${month}/${day}/${year}`;

                        // Checking for Autorización STPS PDF validity
                        const currentDate = new Date();
                        const AutorizacionSTPSDate = new Date(`${month}/${day}/${parseInt(year) + 3}`);

                        result.AutorizacionSTPS = AutorizacionSTPS;
                        result.AutorizacionSTPS.isValid = false;
                        result.AutorizacionSTPS.isDocumentAvailable = true;
                        if (AutorizacionSTPSDate.getTime() > currentDate.getTime() && AutorizacionSTPS.RFC == provider.rfc) {
                            result.AutorizacionSTPS.isValid = true;
                        }
                    }
                    result.AutorizacionSTPS.totalSTPS = 10;

                }
            }

            if (data['Declaración Retención ISR']) {
                let DeclaracionRetencionISR = data['Declaración Retención ISR'][0];
                result.DeclaracionRetencionISR = {
                    RFC: DeclaracionRetencionISR.RFC,
                    EJERCICIO: DeclaracionRetencionISR.Ejercicio,
                    PERIODO: DeclaracionRetencionISR['Período de la declaración']
                };
                result.DeclaracionRetencionISR.isValid = true;
                result.DeclaracionRetencionISR.isDocumentAvailable = true;

            }

            if (data['Pago Retención ISR']) {
                let PagoRetencionISR = mergeValues(data['Pago Retención ISR']);
                let cargoSum = 0;
                result.PagoRetencionISR = {
                    RFC: PagoRetencionISR.RFC,
                    Ejercicio: PagoRetencionISR.Ejercicio,
                    Periodo: PagoRetencionISR['Período de la declaración'],
                    cantidad_a_cargo: PagoRetencionISR.cantidad_a_cargo
                };
                result.PagoRetencionISR.isValid = true;

                // PagoRetencionISR.cantidad_a_cargo.forEach(row => cargoSum += parseFloat(row['Cantidad a pagar'].replaceAll(',', '')));

                // if(result.DeclaracionRetencionISR) {
                //     result.DeclaracionRetencionISR.cargo = cargoSum;
                //     result.DeclaracionRetencionISR.isValid = true;
                // }
                if (result.DeclaracionRetencionISR) {
                    result.DeclaracionRetencionISR.isDocumentAvailable = true;
                }
                if (result.PagoRetencionISR) {
                    result.PagoRetencionISR.isDocumentAvailable = true;
                }
            }

            if (data['Retención IVA']) {
                let RetencionIVA = mergeValues(data['Retención IVA']);

                result.RetencionIVA = {
                    RFC: RetencionIVA.RFC,
                    EJERCICIO: RetencionIVA.EJERCICIO,
                    PERIODO: RetencionIVA.PERIODO,
                    cantidad_a_fav: RetencionIVA['CANTIDAD A FAVOR'],
                    cantidad_a_pag: RetencionIVA['CANTIDAD A PAGAR']
                };
                result.RetencionIVA.isValid = true;
                result.RetencionIVA.isDocumentAvailable = true;

            }

            if (data['Pago IVA']) {
                let PagoIVA = mergeValues(data['Pago IVA']);

                result.PagoIVA = {
                    RFC: PagoIVA.RFC,
                    Ejercicio: PagoIVA.Ejercicio,
                    Periodo: PagoIVA['Período de la declaración'],
                    cantidad_a_cargo: PagoIVA.cantidad_a_cargo,
                    isValid: true
                };

                result.PagoIVA.isDocumentAvailable = true;
            }

            if (data['SUA IMSS'].length > 0) {
                let SUA_IMSS = mergeValues(data['SUA IMSS'][0]);

                if (SUA_IMSS["Período de Proceso"]) {
                    let dateArr = SUA_IMSS["Período de Proceso"].split('-');
                    SUA_IMSS.Periodo = dateArr[0];
                    SUA_IMSS.Ejercicio = dateArr[1];
                } else if (SUA_IMSS["Bimestre de Proceso"]) {
                    let dateArr = SUA_IMSS["Bimestre de Proceso"].split('-');
                    SUA_IMSS.Periodo = dateArr[0];
                    SUA_IMSS.Ejercicio = dateArr[1];
                } else {
                    SUA_IMSS.Periodo = " ";
                    SUA_IMSS.Ejercicio = " ";
                }

                result.SUA_IMSS = {};
                result.SUA_IMSS.RFC = SUA_IMSS.RFC;
                result.SUA_IMSS.Periodo = SUA_IMSS.Periodo;
                result.SUA_IMSS.Ejercicio = SUA_IMSS.Ejercicio;
                result.SUA_IMSS.isValid = false;
                // Combine all PDF pages data in single text to check CURP
                let PDFsCombinedTextArray = await extractAllPDFsPagesText(data['SUA IMSS']);
                // All PDF locations
                result.SUA_IMSS.locations = PDFsCombinedTextArray.map((textObj) => textObj.name);
                // Checking all CURPS include in PDF
                let numOfCurpsFound = 0;
                let checkForCurps = data.CFDIs.map((row) => {
                    let check = PDFsCombinedTextArray.some((TextObj) => {
                        return TextObj.combinedText.includes(row.CURP);
                    });

                    if (check) numOfCurpsFound++;
                });
                // Wait till loop complete
                const resolvePromise = await Promise.all(checkForCurps);
                CheckAllCurpsInclude = (numOfCurpsFound == data.CFDIs.length);
                result.SUA_IMSS.Comentario = `${numOfCurpsFound}/${data.CFDIs.length} TRABAJADORES ENCONTRADOS`;

                if (result.SUA_IMSS.RFC == provider.rfc && CheckAllCurpsInclude) result.SUA_IMSS.isValid = true;
                result.SUA_IMSS.isDocumentAvailable = true;
            }

            if (data['Opinión SAT']) {
                let OpinionSAT = mergeValues(data['Opinión SAT']);

                result.OpinionSAT = OpinionSAT;
                result.OpinionSAT.isValid = false;

                result.OpinionSAT.Estatus = (result.OpinionSAT.POSITIVO) ? 'Positiva' : 'Negativa';
                result.OpinionSAT.Periodo = (result.OpinionSAT.dateText.includes(monthName) || result.OpinionSAT.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionSAT.Ejercicio = result.OpinionSAT.dateText.includes(year) ? year : " ";

                if (OpinionSAT["Clave de R.F.C."] == provider.rfc && result.OpinionSAT.POSITIVO) result.OpinionSAT.isValid = true;
                result.OpinionSAT.isDocumentAvailable = true;
            }

            if (data['Opinión IMSS']) {
                let OpinionIMSS = mergeValues(data['Opinión IMSS']);

                result.OpinionIMSS = OpinionIMSS;
                result.OpinionIMSS.isValid = false;

                result.OpinionIMSS.Estatus = (result.OpinionIMSS.opinionIsPositive) ? 'Positiva' : 'Negativa';
                result.OpinionIMSS.Periodo = (result.OpinionIMSS.dateText.includes(monthName) || result.OpinionIMSS.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionIMSS.Ejercicio = result.OpinionIMSS.dateText.includes(year) ? year : " ";

                if (OpinionIMSS["Clave de R.F.C."] == provider.rfc && result.OpinionIMSS.opinionIsPositive) {
                    result.OpinionIMSS.isValid = true;
                }
                result.OpinionIMSS.isDocumentAvailable = true;
            }

            if (data['Opinión INFONAVIT']) {
                let OpinionINFONAVIT = mergeValues(data['Opinión INFONAVIT']);

                result.OpinionINFONAVIT = OpinionINFONAVIT;
                result.OpinionINFONAVIT.isValid = false;

                result.OpinionINFONAVIT.Estatus = (result.OpinionINFONAVIT.opinionIsPositive) ? 'Positiva' : 'Negativa';
                result.OpinionINFONAVIT.Periodo = (result.OpinionINFONAVIT.dateText.includes(monthName) || result.OpinionINFONAVIT.dateText.includes(monthNameLower)) ? monthName : " ";
                result.OpinionINFONAVIT.Ejercicio = result.OpinionINFONAVIT.dateText.includes(year) ? year : " ";

                if (result.OpinionINFONAVIT.opinionIsPositive) result.OpinionINFONAVIT.isValid = true;
                result.OpinionINFONAVIT.isDocumentAvailable = true;
            }

            if (data['Resumen de Liquidación SUA 1st vers'] || data['Resumen de Liquidación SUA 2nd vers']) {
                resumendeLiquidacionSUA = {
                    isValid: false,
                };
                let firstTotal = 0, secondTotal = 0;

                if (data['Resumen de Liquidación SUA 1st vers']) {
                    let firstData = data['Resumen de Liquidación SUA 1st vers'][0];
                    resumendeLiquidacionSUA['razon_social'] = firstData.razon_social;
                    resumendeLiquidacionSUA['registro_patronal'] = firstData['Registro Patronal'];
                    firstTotal = parseFloat(firstData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                if (data['Resumen de Liquidación SUA 2nd vers']) {
                    let secondData = data['Resumen de Liquidación SUA 2nd vers'][0];
                    resumendeLiquidacionSUA['razon_social'] = secondData.razon_social;
                    resumendeLiquidacionSUA['registro_patronal'] = secondData['Registro Patronal'];
                    secondTotal = parseFloat(secondData['total_a_pagar'].replace(/[$,]/g, ''));
                }

                resumendeLiquidacionSUA.total_a_pagar = (firstTotal >= secondTotal) ? firstTotal : secondTotal;

                if (firstTotal && secondTotal && (firstTotal == secondTotal)) resumendeLiquidacionSUA.isValid = true;

                result.resumendeLiquidacionSUA = resumendeLiquidacionSUA;
                result.resumendeLiquidacionSUA.isDocumentAvailable = true;
            }

            if (data['PAGO IMSS']) {
                let PagoImss = mergeValues(data['PAGO IMSS']);

                if (PagoImss['ano']) {
                    PagoImss['Fecha de Pago'] = (meses.find(_mes => _mes.id == parseInt(PagoImss['mes'])).nombre) + ' ' + PagoImss['ano'];
                }

                result.PagoImss = {
                    razon_social: PagoImss['Razón Social'],
                    fecha_de_pago: (PagoImss['Fecha de Pago']) ? PagoImss['Fecha de Pago'] : PagoImss['Periodo de Pago'],
                    total_de_pago: (PagoImss['Total de Pago']) ? PagoImss['Total de Pago'].replace(/[$,]/g, '') : '',
                    isValid: false
                };

                if (result.resumendeLiquidacionSUA && result.resumendeLiquidacionSUA.total_a_pagar == result.PagoImss.total_de_pago) result.PagoImss.isValid = true;
                result.PagoImss.isDocumentAvailable = true;
            }
            resolve(result);
        } catch (e) {
            console.log(e);
            reject(false);
        }
    });
}


module.exports = {
    deleteInformationFromUserModel: deleteInformationFromUserModel,
    sendOTPinMail: sendOTPinMail,
    sendResetOTPinMail: sendResetOTPinMail,
    createNewDirectory: createNewDirectory,
    uploadFilesTos3: uploadFilesTos3,
    getFilesFromS3: getFilesFromS3,
    DocumentCreation: DocumentCreation,
    ValidateDocumentTypes: ValidateDocumentTypes,
    checkIfDirectoryBelongsToUser: checkIfDirectoryBelongsToUser,
    findIndexOnObject: findIndexOnObject,
    cleanUpDocumentData: cleanUpDocumentData,
    CFDIJSONFormatHelper: CFDIJSONFormatHelper,
    ParseXMLFromDocument: ParseXMLFromDocument,
    SendRegistrationLinkViaEmail: SendRegistrationLinkViaEmail,
    checkDisperancy: checkDisperancy,
    addYearsToDirectory: addYearsToDirectory,
    addMonthsToDirectory: addMonthsToDirectory,
    CountTotalAccount: CountTotalAccount,
    CountTotalUploadedFiles: CountTotalUploadedFiles,
    parsePDFDataFromDocument: parsePDFDataFromDocument,
    uploadFiles: uploadFiles,
    getClientName: getClientName,
    SaveZipDocumentDetail: saveZipDocumentDetail,
    updateZipDocumentStatus: updateZipDocumentStatus,
    SaveZipExtractedDocumentDetail: saveZipExtractedDocumentDetail,
    extractInformationFromDocument: extractInformationFromDocument,
    updateZipExtractedDocumentStatus: updateZipExtractedDocumentStatus,
    DirectoryUpdation: DirectoryUpdation,
    checkAutorizacionSTPSDocStatus: checkAutorizacionSTPSDocStatus,
    checkDirectoryDataStatus: checkDirectoryDataStatus,
    chunkArray: chunkArray,
    processDocumentsArray: processDocumentsArray,
    insertDocumentIdsToDirectory: insertDocumentIdsToDirectory,
    MonthFolderLockstatus: MonthFolderLockstatus,
    uploadClientFiles: uploadClientFiles,
    deleteClientFacturasFiles: deleteClientFacturasFiles,
    AddNotification: AddNotification,
    getNotificationTime: getNotificationTime,
    getdirectoryDetailsbyProvider: getdirectoryDetailsbyProvider,
    ReportDirectoryStatus: ReportDirectoryStatus,
    ReportFolderStatusByComparingDirectoryData: ReportFolderStatusByComparingDirectoryData,
    checkreportFromDocument: checkreportFromDocument,
    generateReportData: generateReportData
}