const objectId = require('mongoose').Types.ObjectId;
const accountsRelations = require('../models/accountsRelations');
const Document = require('../models/documents');
const ZipDocument = require('../models/zipDocument');
const ZipExtractedDocument = require('../models/zipExtractedDocument');
const User = require('../models/user');
const {cleanUpUpload} = require('../../config/uploads');
const {getFilesFromS3,ValidateDocumentTypes,uploadFiles, SaveZipDocumentDetail, updateZipDocumentStatus, SaveZipExtractedDocumentDetail, chunkArray, processDocumentsArray, insertDocumentIdsToDirectory,AddNotification} = require('../helper');
const { v4: uuidv4 } = require('uuid');
const jszip = require('jszip');
const fs = require('fs');
const { exec } = require('child_process');

/* Document Function. */

module.exports = {
    uploadDocument : async (req,res)=>{
        try{

            const {year,month,directoryId,documentType,subDocumentType,isRecurring} = req.body;
            let files = req.file.file;
            if (!Array.isArray(files)) {
                files = [files];
            }
            let check = (isRecurring.toLowerCase() == "true") ? (files && year && month) : files;

            if(check && directoryId && objectId.isValid(directoryId) && documentType && ValidateDocumentTypes(documentType,isRecurring)){

                // Fetch Directory Information (Accounts linked)
                const accountsInformationForDirectory = await accountsRelations.findOne({directory : objectId(directoryId)}).lean();
                const user = await User.findOne({_id: objectId(req.user._id)}).lean();

                if(!(accountsInformationForDirectory.clientId.toString() == req.user._id || accountsInformationForDirectory.providerId.toString() == req.user._id || (accountsInformationForDirectory.clientId.equals( user.clientId ) && user.roles == 'subclient'))){
                    res.status(400);
                    return res.send({Error: "Coulnd't upload your file, Directory Doesn't belong to a user"});
                }

                const uploadAllFiles = uploadFiles(files, year, month, isRecurring, directoryId, documentType, subDocumentType).then((data) => {
                    if(isRecurring.toLowerCase() == "true"){
                    AddNotification(directoryId,year,month,req.user._id,"fileupload");
                    }
                    res.status(200);
                    return res.send({response: uploadAllFiles});
                }).catch((e) => {
                    res.status(400);
                    return res.send({Error: (e) ? e :"El documento que intentas subir es incorrecto, favor de validar su fecha, RFC y formato"});
                });

                // if (uploadAllFiles) {
                //     res.status(200);
                //     return res.send({response: uploadAllFiles});
                // } else {
                //     res.status(400);
                //     return res.send({Error: "Coulnd't upload your file, Please Check file and Try Again"});
                // }
            }else{
                res.status(400);
                return res.send({Error: 'Something went wrong, Incorrect Params'});
            }
        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: 'Something went wrong, Could\'nt process your request'});
        }

    },
    getDocumentById: async(req,res) => {
        try{
            const {id} = req.params;

            if(!id){
                res.status(400);
                return res.send({Error: "Id is not passed as param"});
            }

            const documentRequested = await Document.findOne({uniqueId: id, isValid:true}).lean();

            if(!documentRequested){
                res.status(400);
                return res.send({Error: "Error Fetching your Document"});
            }
            const documentURL = await getFilesFromS3(documentRequested.s3Path, documentRequested.contentType);

            if(!documentURL){
                res.status(400);
                return res.send({Error: "Error Fetching your File"});
            }

            res.status(200);
            return res.send({response: documentURL});

        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: "Something Went wrong, Please Try Again"});
        }
    },
    deleteDocumentById: async (req,res) => {
        try{
            const {id} = req.params;
            if(id){
                const documentRequested = await Document.findOne({uniqueId: id});

                if(!documentRequested){
                    res.status(400);
                    return res.send({Error: "Error Fetching your Document"});
                }

                documentRequested.isValid = false;
                documentRequested.save();

                // Send the Response Back
                res.status(200);
                return res.send({Success: `Document Id ${id} has been successfully deleted.`});

            }else{
                res.status(400);
                return res.send({Error: 'Something went wrong, Incorrect Params'});
            }
        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: "Something Went wrong, Please Try Again"});
        }
    },
    deleteSelectedDocumentByIds: async (req,res) => {
        try{
            const {document_ids} = req.body;
            if(Array.isArray(document_ids) && document_ids.length > 0){
                let promises = document_ids.map(async (id) => {
                    const documentRequested = await Document.findOne({uniqueId: id});

                    if (documentRequested) {
                        documentRequested.isValid = false;
                        documentRequested.save();
                    }
                });

                // wait for delete all documents
                const execAll = await Promise.all(promises);
                // Send the Response Back
                res.status(200);
                return res.send({Success: `All documents has been successfully deleted.`});

            }else{
                res.status(400);
                return res.send({Error: 'Something went wrong, Incorrect Params'});
            }
        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: "Something Went wrong, Please Try Again"});
        }
    },
    uploadZipDocument : async (req,res)=>{
        try{

            const {year,month,directoryId,documentType,isRecurring} = req.body;

            const file = req.file.file;

            if(file && year && month && directoryId && objectId.isValid(directoryId) && documentType && ValidateDocumentTypes(documentType,isRecurring)){

                // Fetch Directory Information (Accounts linked)
                const accountsInformationForDirectory = await accountsRelations.findOne({directory : objectId(directoryId)}).lean();

                if(!(accountsInformationForDirectory.clientId.toString() == req.user._id || accountsInformationForDirectory.providerId.toString() == req.user._id)){
                    res.status(400);
                    return res.send({Error: "Coulnd't upload your file, Directory Doesn't belong to a user"});
                }

                const fileInformation =  {
                    path: file.path
                }; 

                const saveDetail = await SaveZipDocumentDetail(year,month,directoryId,documentType,isRecurring,fileInformation);
                if(saveDetail.isRecurring && saveDetail.documentType && saveDetail.month && saveDetail.year){
                    AddNotification(directoryId,year,month,req.user._id,"fileupload");
                    }
               
                res.status(200);
                return res.send({response: saveDetail});
            }else{
                res.status(400);
                return res.send({Error: 'Something went wrong, Incorrect Params'});
            }
        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: 'Something went wrong, Could\'nt process your request'});
        }

    },
    extractZipDocument : async (req,res)=>{
        try{
            // Fetch Unextracted Zip Document Information
            const zipDoc = await ZipDocument.findOne({isExtracted : false}).lean();

            if (!zipDoc) {
                res.status(400);
                return res.send({Error: 'No zip documents found'});
            }

            const fileContent = fs.readFileSync(zipDoc.path);

            const jszipInstance = new jszip();

            const result = await jszipInstance.loadAsync(fileContent);

            const keys = Object.keys(result.files);
            // console.log(result);

            const folderName = `uploads/extractedDocuments/${zipDoc.directory}/`;
            if (!fs.existsSync(folderName)) {
                fs.mkdirSync(folderName);
            }
            const extractFile = keys.map(async (key) => {
                const documentUniqueId = uuidv4();
                const item = result.files[key];
                let splitName = item.name.split("/");
                let fileName = splitName[splitName.length - 1].replace(/\.[^/.]+$/, "");
                let filePath = folderName+''+documentUniqueId+'_'+fileName;
                if (!item.dir) {

                    if ((item.name.toLowerCase()).includes('.xml')) {
                        if (!fs.existsSync(filePath)) {
                            fs.writeFileSync(filePath, Buffer.from(await item.async('arraybuffer')));
                            const saveExtractedFileDetail = await SaveZipExtractedDocumentDetail(zipDoc.year, zipDoc.month, zipDoc.directory, zipDoc._id, zipDoc.documentType, zipDoc.isRecurring, filePath, fileName);
                        }
                    }
                }
            });

            await Promise.all(extractFile);


            const execCommand = `aws s3 sync ./uploads/extractedDocuments/${zipDoc.directory}/ s3://ebitsa/${zipDoc.directory}/Documentacion_Mensual/${zipDoc.year}/${zipDoc.month}/cfdis/`;


            // Execute command to sync files to AWS S3
            exec(execCommand, (e, stdout, stderr) => {
                console.log('stdout', stdout);
                console.log('stderr', stderr);
            });

            const updateStatus = await updateZipDocumentStatus(zipDoc._id);

            // Cleanup Locally
            cleanUpUpload(zipDoc.path);

            res.status(200);
            return res.send({response: "Zip extracted successfully"});
        }catch(e){
            console.log(e);
            res.status(400);
            return res.send({Error: 'Something went wrong, Could\'nt process your request'});
        }

    },

    extractDataFromZipDocument : async (req,res)=>{
        try {
            const documentDetails = await ZipExtractedDocument.find({isDataExtracted: false, isSelected: false}).limit(1000).lean();

            if (!documentDetails.length) {
                res.status(400);
                return res.send({Error: 'No documents found'});
            }

            const bulkOps = documentDetails.map(obj => {
                return {
                  updateOne: {
                    filter: {
                      _id: obj._id
                    },
                    update: {
                        $set: {
                            isSelected: true
                        }
                    }
                  }
                }
            });
            const update = await ZipExtractedDocument.bulkWrite(bulkOps);

            // Split documents into multiple chunks
            const chunkedArray = await chunkArray(documentDetails, 250);

            let processedDocumentsIdsArr = [];
            // process chunked array documents one by one
            const processChunkedArray = chunkedArray.map(async (documentDetailsArray) => {
                const processedDocumentsIds = await processDocumentsArray(documentDetailsArray);
                processedDocumentsIdsArr.push(processedDocumentsIds);
            });

            await Promise.all(processChunkedArray);

            let directoryDocumentIds = {};
            //process documentIds array
            const DocumentsIdsArr = processedDocumentsIdsArr.map(async (DocumentsIdsObj) => {
                const combineIds = Object.keys(DocumentsIdsObj).map((key) => {
                    if(key in directoryDocumentIds){
                        directoryDocumentIds[key].documentIds = directoryDocumentIds[key].documentIds.concat(DocumentsIdsObj[key].documentIds);

                    } else { 
                        directoryDocumentIds[key] = DocumentsIdsObj[key];
                    }
                });

                await Promise.all(combineIds);
            });

            await Promise.all(DocumentsIdsArr);
            const insertIdsToDirectory = await insertDocumentIdsToDirectory(directoryDocumentIds);

            // console.log(documentDetails);
            res.status(200);
            return res.send({response: 'Data extracted from documents'});
        } catch (e) {
            console.log(e);
            res.status(400);
            return res.send({Error: 'Something went wrong, Could\'nt process your request'});
        }
    },

    getZipExtractedDocuments: async (req,res)=>{
        try {
            const {directory,year,month,folder_name} = req.body;
            const documentDetails = await ZipExtractedDocument.find({directory: directory, year: year, month: month, folder_name: folder_name}).lean();

            res.status(200);
            return res.send({data: documentDetails});
        } catch (e) {
            res.status(400);
            return res.send({Error: 'Something went wrong, Could\'nt process your request'});
        }
    }
}